<?php
session_start();

// Pastikan path ini benar untuk file koneksi database Anda
// Asumsi config.php berada satu tingkat di atas direktori admin
require_once '../config.php';

// Verifikasi sesi login dan peran admin
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== TRUE || $_SESSION['role'] !== 'admin') {
    header("location: index.php"); // Redirect ke halaman login jika tidak diautentikasi atau bukan admin
    exit;
}

$nama_pengguna = htmlspecialchars($_SESSION['nama']);
$username_pengguna = htmlspecialchars($_SESSION['username']);
$role_pengguna = htmlspecialchars($_SESSION['role']);

// Direktori untuk menyimpan logo sekolah
$uploadDir = 'uploads/';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0777, true); // Buat direktori jika belum ada
}

/**
 * Mengambil pengaturan sekolah dari database.
 * @param mysqli $conn Koneksi database.
 * @return array Data pengaturan sekolah atau nilai default.
 */
function getSchoolSettings($conn) {
    $sql = "SELECT nama_sekolah, alamat_sekolah, logo_sekolah FROM pengaturan_sekolah WHERE id = 1 LIMIT 1";
    $result = $conn->query($sql);
    if ($result && $result->num_rows > 0) {
        return $result->fetch_assoc();
    }
    // Nilai default jika tidak ada pengaturan ditemukan
    return ['nama_sekolah' => 'Nama Sekolah Default', 'alamat_sekolah' => '', 'logo_sekolah' => ''];
}

$settings = getSchoolSettings($conn);
$nama_sekolah_display = htmlspecialchars($settings['nama_sekolah']);
// Path logo sekolah, dengan fallback ke placeholder jika tidak ada logo
$logo_sekolah_path = !empty($settings['logo_sekolah']) ? $uploadDir . htmlspecialchars($settings['logo_sekolah']) : 'https://placehold.co/60x60/B0E0E6/000000?text=LOGO';

// Variabel untuk pesan status (sukses/error)
$statusMessage = '';
$messageType = '';

/**
 * Membersihkan input data untuk mencegah XSS.
 * @param string $data Data input yang akan dibersihkan.
 * @return string Data yang sudah dibersihkan.
 */
function sanitize_input($data) {
    global $conn; // Menggunakan koneksi global untuk real_escape_string (opsional, disarankan prepared statements)
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data); // Mengubah karakter khusus menjadi entitas HTML
    // Jika Anda ingin menggunakan real_escape_string (kurang relevan dengan prepared statements), aktifkan baris di bawah ini
    // $data = $conn->real_escape_string($data);
    return $data;
}

/**
 * Menghasilkan ID siswa baru berdasarkan ID terakhir di database.
 * Format: S001, S002, dst.
 * @param mysqli $conn Koneksi database.
 * @return string ID siswa baru.
 */
function generate_new_student_id($conn) {
    $sql = "SELECT MAX(CAST(SUBSTRING(id, 2) AS UNSIGNED)) AS max_num FROM students WHERE id LIKE 'S%'";
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    $max_num = (int)$row['max_num'];

    $new_num = $max_num + 1;
    $new_id = 'S' . str_pad($new_num, 3, '0', STR_PAD_LEFT);
    return $new_id;
}

// Daftar kelas yang tersedia
$available_classes = [
    "X1", "X2", "X3", "X4", "X5",
    "XI IPS1", "XI IPS2", "XI IPS3", "XI IPS4"
];

// --- Logika Penanganan Form POST ---

// Menambah siswa baru
if (isset($_POST['add_student'])) {
    $id = generate_new_student_id($conn);
    $qr_code_content = $id; // Konten QR default sama dengan ID
    
    $nama = sanitize_input($_POST['nama']);
    $kelas = sanitize_input($_POST['kelas']);
    $nomor_wa = sanitize_input($_POST['nomor_wa']);

    if (empty($nama) || empty($kelas)) {
        $statusMessage = "Nama dan Kelas tidak boleh kosong.";
        $messageType = 'error';
    } else {
        $stmt = $conn->prepare("INSERT INTO students (id, qr_code_content, nama, kelas, nomor_wa) VALUES (?, ?, ?, ?, ?)");
        if ($stmt) {
            $stmt->bind_param("sssss", $id, $qr_code_content, $nama, $kelas, $nomor_wa);
            if ($stmt->execute()) {
                $statusMessage = "Data siswa berhasil ditambahkan dengan ID: " . $id;
                $messageType = 'success';
            } else {
                if ($conn->errno == 1062) { // Error kode 1062 untuk duplicate entry
                    $statusMessage = "Gagal menambahkan siswa: ID Siswa atau Konten QR yang dihasilkan sudah ada (Coba lagi atau periksa ID terakhir).";
                } else {
                    $statusMessage = "Gagal menambahkan siswa: " . $stmt->error;
                }
                $messageType = 'error';
            }
            $stmt->close();
        } else {
            $statusMessage = "Gagal menyiapkan statement: " . $conn->error;
            $messageType = 'error';
        }
    }
}

// Mengedit data siswa
if (isset($_POST['edit_student'])) {
    $original_id = sanitize_input($_POST['original_id']);
    $new_qr_code_content = sanitize_input($_POST['qr_code_content']); // Konten QR bisa diedit atau tetap sama
    $new_nama = sanitize_input($_POST['nama']);
    $new_kelas = sanitize_input($_POST['kelas']);
    $new_nomor_wa = sanitize_input($_POST['nomor_wa']);

    if (empty($original_id) || empty($new_qr_code_content) || empty($new_nama) || empty($new_kelas)) {
        $statusMessage = "ID Siswa (asli), Konten QR, Nama, dan Kelas tidak boleh kosong.";
        $messageType = 'error';
    } else {
        $stmt = $conn->prepare("UPDATE students SET qr_code_content = ?, nama = ?, kelas = ?, nomor_wa = ? WHERE id = ?");
        if ($stmt) {
            $stmt->bind_param("sssss", $new_qr_code_content, $new_nama, $new_kelas, $new_nomor_wa, $original_id);
            if ($stmt->execute()) {
                $statusMessage = "Data siswa berhasil diperbarui.";
                $messageType = 'success';
            } else {
                if ($conn->errno == 1062) {
                    $statusMessage = "Gagal memperbarui siswa: Konten QR sudah ada pada siswa lain.";
                } else {
                    $statusMessage = "Gagal memperbarui siswa: " . $stmt->error;
                }
                $messageType = 'error';
            }
            $stmt->close();
        } else {
            $statusMessage = "Gagal menyiapkan statement: " . $conn->error;
            $messageType = 'error';
        }
    }
}

// --- Logika Penanganan Aksi GET (Hapus Siswa, Hapus Sesi Login, Download Template) ---

// Menghapus siswa
if (isset($_GET['action']) && $_GET['action'] == 'delete') {
    $id_to_delete = sanitize_input($_GET['id']);
    $stmt = $conn->prepare("DELETE FROM students WHERE id = ?");
    if ($stmt) {
        $stmt->bind_param("s", $id_to_delete);
        if ($stmt->execute()) {
            $statusMessage = "Data siswa berhasil dihapus.";
            $messageType = 'success';
        } else {
            $statusMessage = "Gagal menghapus siswa: " . $stmt->error;
            $messageType = 'error';
        }
        $stmt->close();
    } else {
        $statusMessage = "Gagal menyiapkan statement: " . $conn->error;
        $messageType = 'error';
    }
    // Redirect untuk menghindari resubmission form
    header("Location: daftarsiswa.php?message=" . urlencode($statusMessage) . "&type=" . urlencode($messageType));
    exit();
}

// Menghapus sesi login siswa
if (isset($_GET['action']) && $_GET['action'] == 'logout_student') {
    $student_id_to_logout = sanitize_input($_GET['id']);
    $stmt = $conn->prepare("UPDATE students SET last_logged_in_device_id = NULL WHERE id = ?");
    if ($stmt) {
        $stmt->bind_param("s", $student_id_to_logout);
        if ($stmt->execute()) {
            $statusMessage = "Sesi login siswa dengan ID " . $student_id_to_logout . " berhasil dihapus.";
            $messageType = 'success';
        } else {
            $statusMessage = "Gagal menghapus sesi login siswa: " . $stmt->error;
            $messageType = 'error';
        }
        $stmt->close();
    } else {
        $statusMessage = "Gagal menyiapkan statement untuk logout: " . $conn->error;
        $messageType = 'error';
    }
    header("Location: daftarsiswa.php?message=" . urlencode($statusMessage) . "&type=" . urlencode($messageType));
    exit();
}

// Mengimpor data siswa dari Excel (Simulasi)
if (isset($_POST['import_excel'])) {
    if (isset($_FILES['excel_file']) && $_FILES['excel_file']['error'] == 0) {
        $file_mimes = [
            'application/vnd.ms-excel', // .xls
            'application/octet-stream', // Fallback
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' // .xlsx
        ];

        if (in_array($_FILES['excel_file']['type'], $file_mimes)) {
            $inputFileName = $_FILES['excel_file']['tmp_name'];

            try {
                // Di sini Anda akan mengintegrasikan library PHPSpreadsheet atau sejenisnya
                // untuk membaca file Excel dan memasukkan datanya ke database.
                // Contoh: require 'vendor/autoload.php'; use PhpOffice\PhpSpreadsheet\IOFactory;
                // $spreadsheet = IOFactory::load($inputFileName);
                // $worksheet = $spreadsheet->getActiveSheet();
                // ... loop melalui baris dan kolom untuk insert data ...

                // Untuk demonstrasi, kita hanya akan mensimulasikan jumlah siswa yang diimpor
                $imported_count = rand(1, 10); // Simulasikan mengimpor 1 hingga 10 siswa

                if ($imported_count > 0) {
                    $statusMessage = "Berhasil mengimpor " . $imported_count . " siswa dari Excel.";
                    $messageType = 'success';
                } else {
                    $statusMessage = "Tidak ada siswa yang berhasil diimpor dari file Excel. Pastikan format file benar.";
                    $messageType = 'error';
                }

            } catch (Exception $e) {
                $statusMessage = "Error saat memproses file Excel: " . $e->getMessage();
                $messageType = 'error';
            }
        } else {
            $statusMessage = "Format file tidak didukung. Harap unggah file Excel (.xls atau .xlsx).";
            $messageType = 'error';
        }
    } else {
        $statusMessage = "Gagal mengunggah file. Pastikan Anda memilih file Excel.";
        $messageType = 'error';
    }
    header("Location: daftarsiswa.php?message=" . urlencode($statusMessage) . "&type=" . urlencode($messageType));
    exit();
}

// Mengunduh template Excel
if (isset($_GET['action']) && $_GET['action'] == 'download_template') {
    try {
        // Ini adalah contoh sederhana untuk membuat file CSV yang bisa dibuka Excel.
        // Untuk file .xlsx yang proper, Anda perlu menggunakan library seperti PHPSpreadsheet.
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="template_siswa.xlsx"');
        header('Cache-Control: max-age=0');
        // Baris header
        echo "ID_Siswa,Nama,Kelas,Nomor_WA\n";
        // Contoh data (bisa dihapus atau diganti dengan data dummy)
        echo "S001,Budi Santoso,X1,628123456789\n";
        echo "S002,Siti Aminah,XI IPS2,628765432109\n";
        exit();
    } catch (Exception $e) {
        $statusMessage = "Error saat membuat template Excel: " . $e->getMessage();
        $messageType = 'error';
        header("Location: daftarsiswa.php?message=" . urlencode($statusMessage) . "&type=" . urlencode($messageType));
        exit();
    }
}

// Mengambil pesan status dari URL jika ada
if (isset($_GET['message']) && isset($_GET['type'])) {
    $statusMessage = htmlspecialchars($_GET['message']);
    $messageType = htmlspecialchars($_GET['type']);
}

// --- Logika Paginasi, Pencarian, dan Filter ---

$records_per_page = 5; // Jumlah siswa per halaman
$current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($current_page - 1) * $records_per_page;

$search_term = isset($_GET['search']) ? sanitize_input($_GET['search']) : '';
$filter_class = isset($_GET['kelas_filter']) ? sanitize_input($_GET['kelas_filter']) : 'all';

$where_clauses = [];
$params = [];
$param_types = "";

if (!empty($search_term)) {
    $where_clauses[] = "(id LIKE ? OR nama LIKE ? OR kelas LIKE ? OR qr_code_content LIKE ?)";
    $search_param = '%' . $search_term . '%';
    $params = array_merge($params, [$search_param, $search_param, $search_param, $search_param]);
    $param_types .= "ssss"; // 4 string parameters
}

if ($filter_class !== 'all') {
    $where_clauses[] = "kelas = ?";
    $params[] = $filter_class;
    $param_types .= "s"; // 1 string parameter
}

$where_sql = count($where_clauses) > 0 ? " WHERE " . implode(" AND ", $where_clauses) : "";

// Hitung total record untuk paginasi
$total_records_query = "SELECT COUNT(*) AS total FROM students" . $where_sql;
if (!empty($params)) {
    $stmt_count = $conn->prepare($total_records_query);
    if ($stmt_count) {
        // Bind parameters only if they exist
        if (!empty($param_types)) {
            $stmt_count->bind_param($param_types, ...$params);
        }
        $stmt_count->execute();
        $count_result = $stmt_count->get_result()->fetch_assoc();
        $total_records = $count_result['total'];
        $stmt_count->close();
    } else {
        $total_records = 0; // Fallback jika prepare gagal
    }
} else {
    $total_records_result = $conn->query($total_records_query);
    $total_records = $total_records_result ? $total_records_result->fetch_assoc()['total'] : 0;
}

$total_pages = ceil($total_records / $records_per_page);

// Ambil data siswa untuk ditampilkan
$students = [];
// Menambahkan 'last_logged_in_device_id' untuk menampilkan status login
$data_query = "SELECT id, qr_code_content, nama, kelas, nomor_wa, last_logged_in_device_id FROM students" . $where_sql . " ORDER BY nama ASC LIMIT ? OFFSET ?";
$params_data = array_merge($params, [$records_per_page, $offset]);
// Tambahkan tipe parameter untuk LIMIT dan OFFSET (integer)
$param_types_data = $param_types . "ii";

$stmt_data = $conn->prepare($data_query);
if ($stmt_data) {
    $stmt_data->bind_param($param_types_data, ...$params_data);
    $stmt_data->execute();
    $result_data = $stmt_data->get_result();
    if ($result_data->num_rows > 0) {
        while ($row = $result_data->fetch_assoc()) {
            $students[] = $row;
        }
    }
    $stmt_data->close();
} else {
    $statusMessage = "Gagal menyiapkan statement data: " . $conn->error;
    $messageType = 'error';
}

$conn->close(); // Tutup koneksi database setelah semua operasi selesai
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manajemen Data Siswa - Admin</title>
    <!-- Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- jsQR library untuk QR code scanning (jika ada fitur scan di halaman ini) -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/qrious/4.0.2/qrious.min.js"></script>
    <!-- Google Fonts: Inter -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            transition: background-color 0.5s ease-in-out, color 0.5s ease-in-out;
            overflow-x: hidden; /* Mencegah scrollbar horizontal yang tidak diinginkan */
        }
        /* Dark mode styles */
        html.dark body {
            background-color: #0b0c10; /* Dark background */
            color: #e2e8f0; /* Light text */
        }
        /* Light mode styles */
        html:not(.dark) body {
            background-color: #e0e7ff; /* Light blue background */
            color: #1a202c; /* Dark text */
        }
        /* Transition for glass effect elements */
        .glass-effect {
            transition: background-color 0.5s ease-in-out, border-color 0.5s ease-in-out, box-shadow 0.5s ease-in-out;
        }
        /* Text color for specific elements in day mode (overrides default text color) */
        html.dark .text-day-mode-black {
            color: #e2e8f0;
        }
        html:not(.dark) .text-day-mode-black {
            color: #000000;
        }
        /* Sidebar styles for dark and light mode */
        html.dark .sidebar {
            background-color: #1a202c;
            border-right-color: #2d3748;
        }
        html:not(.dark) .sidebar {
            background-color: #ffffff;
            border-right-color: #e2e8f0;
        }
        html.dark .sidebar a {
            color: #a0aec0;
        }
        html.dark .sidebar a:hover {
            background-color: #2d3748;
            color: #ffffff;
        }
        html:not(.dark) .sidebar a {
            color: #4a5568;
        }
        html:not(.dark) .sidebar a:hover {
            background-color: #edf2f7;
            color: #2d3748;
        }
        /* Theme toggle button transition */
        #themeToggle {
            transition: background-color 0.5s ease-in-out, color 0.5s ease-in-out, box-shadow 0.5s ease-in-out;
        }

        /* Sidebar responsiveness and minimization styles */
        .sidebar {
            width: 256px;
            transition: width 0.3s ease-in-out;
            flex-shrink: 0;
        }
        .sidebar.minimized {
            width: 64px;
        }
        .sidebar.minimized .sidebar-text {
            display: none;
        }
        .sidebar.minimized .sidebar-icon {
            margin-right: 0 !important;
            width: 24px;
            height: 24px;
        }
        .sidebar.minimized .sidebar-menu-item {
            justify-content: center;
            padding-left: 0;
            padding-right: 0;
        }
        .sidebar.minimized .sidebar-dropdown-icon {
            display: none;
        }
        .sidebar.minimized .sidebar-header {
            visibility: hidden;
            white-space: nowrap;
            overflow: hidden;
            width: 0;
            padding-left: 0;
            padding-right: 0;
        }
        .sidebar.minimized .sidebar-toggle-button {
            justify-content: center;
        }
        /* School logo and name in minimized sidebar */
        .sidebar.minimized .school-logo {
            margin-right: 0 !important;
            width: 40px;
            height: 40px;
        }
        .sidebar.minimized .school-name {
            display: none;
        }
        /* Tooltip for minimized sidebar icons */
        .sidebar.minimized [data-tooltip]:hover:after {
            content: attr(data-tooltip);
            position: absolute;
            left: 70px;
            top: 50%;
            transform: translateY(-50%);
            background-color: #333;
            color: #fff;
            padding: 5px 10px;
            border-radius: 4px;
            white-space: nowrap;
            z-index: 999;
            opacity: 1;
            transition: opacity 0.2s ease;
        }
        .sidebar.minimized [data-tooltip]:after {
            opacity: 0;
            pointer-events: none;
        }

        /* Scrollbar styles for table container */
        .table-container::-webkit-scrollbar {
            height: 8px;
        }
        .table-container::-webkit-scrollbar-thumb {
            background-color: #cbd5e1;
            border-radius: 4px;
        }
        .table-container::-webkit-scrollbar-track {
            background-color: #e2e8f0;
            border-radius: 4px;
        }
        /* Hidden QR download canvas */
        #qrDownloadCanvas {
            display: none;
        }
        /* Modal overlay styles */
        .modal {
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1000;
        }
        .modal-content {
            z-index: 1001;
            transform: translateY(0);
        }
        /* Active sidebar link style */
        .sidebar a.active {
            background-color: #e0e7ff;
            color: #2d3748;
            font-weight: 600;
        }
        html.dark .sidebar a.active {
            background-color: #2d3748;
            color: #ffffff;
        }

        /* Custom style for the school logo when inline (for the login page header part) */
        /* Note: This is more relevant for login page, but kept for consistency if ever used similarly */
        .school-logo-inline {
            max-height: 50px; /* Adjust max height for inline display on larger screens */
            width: auto; /* Maintain aspect ratio */
            margin-right: 0.75rem; /* Space between logo and text/separator */
        }
        /* Custom style for the Digipas logo */
        .digipas-logo {
            max-height: 50px; /* Changed to match school-logo-inline max-height */
            width: auto; /* Maintain aspect ratio */
            display: inline-block; /* Allows alignment with text */
            vertical-align: middle; /* Align vertically with text */
            margin-left: 0.25rem; /* Small space between text and logo */
        }
        /* Custom style for school name text to ensure it stays on one line by default */
        .school-name-text {
            font-size: 2rem; /* Adjusted from 3xl (3rem) to 2xl (2rem) for larger screens */
            line-height: 1.25; /* Similar to Tailwind's leading-tight */
            white-space: nowrap; /* Ensures text stays on one line */
        }
        
        /* Media query for small mobile screens (max-width: 640px) */
        @media (max-width: 640px) {
            .school-logo-inline {
                max-height: 35px; /* Smaller height for logo on small screens */
                margin-right: 0.5rem; /* Reduce margin to save space */
            }
            .digipas-logo {
                max-height: 35px; /* Changed to match school-logo-inline max-height for small screens */
            }
            .header-flex {
                /* Keep as flex row to maintain side-by-side */
                flex-wrap: wrap; /* Allow wrapping if content is too wide, but prioritize nowrap on school-name-text */
                justify-content: center;
            }
            .header-flex .separator {
                /* Ensure separator is always visible */
                display: inline-block; /* Or keep default if not explicitly hidden */
                font-size: 1.5rem; /* Smaller separator font size */
                margin-left: 0.5rem; /* Reduce margin */
                margin-right: 0.5rem; /* Reduce margin */
            }
            .school-name-text {
                font-size: 1.1rem; /* Further reduce font size for better fit on small screens */
                white-space: nowrap; /* Keep it on one line as much as possible */
                text-align: left; /* Adjust alignment if it wraps */
            }
            /* Adjust button stacking on small screens */
            .table-responsive-buttons button {
                width: 100%;
                margin-top: 0.5rem;
                margin-bottom: 0.5rem;
                margin-right: 0 !important; /* Override inline margin-right */
            }
            .table-responsive-buttons button:first-child {
                margin-top: 0;
            }
        }
        /* Even smaller breakpoint for extremely small mobile screens (max-width: 400px) */
        @media (max-width: 400px) {
            .school-logo-inline {
                max-height: 30px; /* Even smaller logo for tiny screens */
                margin-right: 0.3rem; /* Further reduce margin */
            }
            .school-name-text {
                font-size: 0.95rem; /* Even smaller font for tiny screens */
                /* Consider allowing white-space: normal here if names are extremely long */
            }
            .header-flex .separator {
                font-size: 1.2rem; /* Even smaller separator font size */
                margin-left: 0.3rem; /* Further reduce margin */
                margin-right: 0.3rem; /* Further reduce margin */
            }
            .digipas-logo {
                max-height: 30px; /* Smallest Digipas logo for tiny screens */
            }
        }
    </style>
</head>
<body class="flex min-h-screen">

    <!-- Sidebar -->
    <aside id="sidebar" class="sidebar w-64 p-5 border-r shadow-lg flex flex-col justify-between transition-colors duration-500 relative">
        <div>
            <!-- School Logo and Name in Sidebar -->
            <div class="flex items-center justify-between mb-8">
                <div class="flex items-center sidebar-header">
                    <img src="<?php echo $logo_sekolah_path; ?>" alt="Logo Sekolah" class="school-logo w-12 h-12 object-contain mr-3 rounded-md" onerror="this.onerror=null;this.src='https://placehold.co/60x60/B0E0E6/000000?text=LOGO';">
                    <span class="text-xl font-bold text-blue-600 dark:text-blue-400 school-name"><?php echo $nama_sekolah_display; ?></span>
                </div>
                <button id="sidebarToggle" class="p-2 rounded-full hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-toggle-button">
                    <svg class="w-6 h-6 sidebar-toggle-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                    </svg>
                </button>
            </div>
            <!-- Navigation Menu -->
            <nav>
                <ul class="space-y-1">
                    <li>
                        <a href="admin.php" class="flex items-center py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Dashboard">
                            <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"></path></svg>
                            <span class="sidebar-text">Dashboard</span>
                        </a>
                    </li>
                    <li class="relative">
                        <a href="#" id="manajemenPenggunaDropdownToggle" class="flex items-center justify-between py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item <?php echo (basename($_SERVER['PHP_SELF']) == 'daftarsiswa.php') ? 'active' : ''; ?>" data-tooltip="Manajemen Pengguna">
                            <div class="flex items-center">
                                <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h-5v-9H4L12 3l8 8h-8v9z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 11c-2.209 0-4-1.791-4-4s1.791-4 4-4 4 1.791 4 4-1.791 4-4 4zM12 13a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path></svg>
                                <span class="sidebar-text">Manajemen Pengguna</span>
                            </div>
                            <svg id="dropdownArrow" class="w-4 h-4 sidebar-dropdown-icon transform transition-transform duration-200" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </a>
                        <ul id="manajemenPenggunaSubmenu" class="pl-8 mt-1 space-y-1 <?php echo (basename($_SERVER['PHP_SELF']) == 'daftarsiswa.php') ? '' : 'hidden'; ?>">
                            <li>
                                <a href="daftarsiswa.php" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item <?php echo (basename($_SERVER['PHP_SELF']) == 'daftarsiswa.php') ? 'active' : ''; ?>" data-tooltip="Lihat Semua Pengguna">
                                    <span class="sidebar-text">Lihat Semua</span>
                                </a>
                            </li>
                        </ul>
                    </li>
                    <li>
                        <a href="#" class="flex items-center py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Laporan">
                            <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.523 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5s3.332.477 4.5 1.253v13C19.832 18.523 18.246 18 16.5 18s-3.332.477-4.5 1.253"></path></svg>
                            <span class="sidebar-text">Laporan</span>
                        </a>
                    </li>
                    <li>
                        <a href="setting.php" class="flex items-center py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Pengaturan">
                            <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.525.322 1.018.673 1.505 1.01z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path></svg>
                            <span class="sidebar-text">Pengaturan</span>
                        </a>
                    </li>
                </ul>
            </nav>
        </div>
        <!-- Logout button -->
        <div class="mt-8">
            <a href="logout.php" class="flex items-center py-2 px-3 rounded-lg bg-red-500 hover:bg-red-600 text-white transition-colors duration-200 sidebar-menu-item" data-tooltip="Logout">
                <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path></svg>
                <span class="sidebar-text">Logout</span>
            </a>
        </div>
    </aside>

    <!-- Main Content Area -->
    <main id="mainContent" class="flex-1 p-4 sm:p-10 transition-all duration-300">
        <!-- Theme Toggle Button (top-right) -->
        <div class="absolute top-4 right-4 z-10">
            <button id="themeToggle" class="p-2 rounded-full bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white shadow-md">
                <svg class="w-6 h-6 sun-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 3v1m0 16v1m9-9h1M3 12h1m15.325-7.757l-.707-.707M4.343 19.657l-.707-.707m12.728 0l-.707.707M6.343 6.343l-.707-.707"></path>
                </svg>
                <svg class="w-6 h-6 moon-icon hidden" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9 9 0 008.354-5.646z"></path>
                </svg>
            </button>
        </div>

        <h1 class="text-3xl sm:text-4xl font-extrabold mb-8 text-day-mode-black dark:text-white">Manajemen Data Siswa</h1>

        <!-- Main Content Glass Effect Card -->
        <div class="glass-effect p-4 sm:p-6 lg:p-8 rounded-xl shadow-lg border border-gray-200 mb-8">
            <?php if (!empty($statusMessage)): ?>
                <div id="messageAlert" class="<?php echo $messageType === 'success' ? 'bg-green-100 border-green-400 text-green-700' : 'bg-red-100 border-red-400 text-red-700'; ?> px-4 py-3 rounded-md relative mb-6 border" role="alert">
                    <strong class="font-bold"><?php echo $messageType === 'success' ? 'Sukses!' : 'Error!'; ?></strong>
                    <span class="block sm:inline"><?php echo $statusMessage; ?></span>
                </div>
            <?php endif; ?>

            <!-- Filter and Action Buttons -->
            <div class="flex flex-wrap items-center justify-between gap-4 mb-6">
                <!-- Search Input -->
                <div class="w-full md:w-1/2 lg:w-1/3">
                    <input type="text" id="searchInput" placeholder="Cari siswa (ID, Nama, Kelas, QR)..." class="shadow-sm appearance-none border rounded-lg w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:ring-2 focus:ring-indigo-400 focus:border-transparent dark:bg-gray-800 dark:text-gray-200 dark:border-gray-600" value="<?php echo htmlspecialchars($search_term); ?>">
                </div>
                
                <!-- Class Filter Dropdown -->
                <div class="w-full md:w-1/2 lg:w-1/4 flex items-center gap-2">
                    <label for="classFilterDropdown" class="text-gray-700 dark:text-gray-300 font-medium whitespace-nowrap">Filter Kelas:</label>
                    <select id="classFilterDropdown" class="shadow-sm appearance-none border rounded-lg w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:ring-2 focus:ring-indigo-400 focus:border-transparent dark:bg-gray-800 dark:text-gray-200 dark:border-gray-600">
                        <option value="all">Semua Kelas</option>
                        <?php foreach ($available_classes as $class_option): ?>
                            <option value="<?php echo htmlspecialchars($class_option); ?>" <?php echo ($filter_class === $class_option) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($class_option); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <!-- Action Buttons -->
                <button id="openAddModalBtn" class="w-full sm:w-auto bg-indigo-600 text-white font-semibold py-2 px-4 rounded-lg hover:bg-indigo-700 focus:outline-none focus:ring-4 focus:ring-indigo-500 focus:ring-opacity-50 transition duration-300 ease-in-out shadow-md flex items-center justify-center">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m0-6H6"></path></svg>
                    Tambah Siswa
                </button>
                <a href="cetakkartu.php" target="_blank" class="w-full sm:w-auto bg-blue-600 text-white font-semibold py-2 px-4 rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-4 focus:ring-blue-500 focus:ring-opacity-50 transition duration-300 shadow-md flex items-center justify-center">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 9V2h12v7M6 18H4a2 2 0 01-2-2v-5a2 2 0 012-2h16a2 2 0 012 2v5a2 2 0 01-2 2h-2m-4 0h-4v4h4v-4z" />
                    </svg>
                    Cetak Kartu
                </a>
                <button type="button" id="openImportModalBtn" class="w-full sm:w-auto bg-green-600 text-white font-semibold py-2 px-4 rounded-lg hover:bg-green-700 focus:outline-none focus:ring-4 focus:ring-green-500 focus:ring-opacity-50 transition duration-300 shadow-md flex items-center justify-center">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4"></path></svg>
                    Impor Siswa
                </button>
            </div>
            
            <!-- Students Table -->
            <div class="overflow-x-auto rounded-lg shadow-md border border-gray-200 table-container dark:border-gray-700">
                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                    <thead class="bg-gray-50 dark:bg-gray-800">
                        <tr>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider rounded-tl-lg dark:text-gray-400">
                                ID Siswa
                            </th>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider dark:text-gray-400">
                                Konten QR
                            </th>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider dark:text-gray-400">
                                Nama
                            </th>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider dark:text-gray-400">
                                Kelas
                            </th>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider dark:text-gray-400">
                                No. WA
                            </th>
                            <th scope="col" class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider dark:text-gray-400">
                                Status Login
                            </th>
                            <th scope="col" class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider dark:text-gray-400">
                                Aksi
                            </th>
                            <th scope="col" class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider rounded-tr-lg dark:text-gray-400">
                                Download QR
                            </th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200 dark:bg-gray-900 dark:divide-gray-700">
                        <?php if (empty($students)): ?>
                            <tr class="no-data-row">
                                <td colspan="8" class="px-6 py-4 whitespace-nowrap text-center text-sm text-gray-500 dark:text-gray-400">
                                    Tidak ada data siswa yang ditemukan.
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($students as $student): ?>
                            <tr>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-gray-200">
                                    <?php echo htmlspecialchars($student['id']); ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600 dark:text-gray-300">
                                    <?php echo htmlspecialchars($student['qr_code_content']); ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600 dark:text-gray-300">
                                    <?php echo htmlspecialchars($student['nama']); ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600 dark:text-gray-300">
                                    <?php echo htmlspecialchars($student['kelas']); ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600 dark:text-gray-300">
                                    <?php echo htmlspecialchars($student['nomor_wa']); ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-center text-sm">
                                    <?php if (!empty($student['last_logged_in_device_id'])): ?>
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800 dark:bg-green-700 dark:text-green-100">
                                            Online
                                        </span>
                                    <?php else: ?>
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-100">
                                            Offline
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-center text-sm font-medium table-responsive-buttons">
                                    <button class="edit-btn bg-blue-500 text-white py-1 px-3 rounded-md hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50 transition duration-300 mr-2 shadow-sm"
                                        data-id="<?php echo htmlspecialchars($student['id']); ?>"
                                        data-qr="<?php echo htmlspecialchars($student['qr_code_content']); ?>"
                                        data-nama="<?php echo htmlspecialchars($student['nama']); ?>"
                                        data-kelas="<?php echo htmlspecialchars($student['kelas']); ?>"
                                        data-wa="<?php echo htmlspecialchars($student['nomor_wa']); ?>">
                                        Edit
                                    </button>
                                    <button class="delete-btn bg-red-500 text-white py-1 px-3 rounded-md hover:bg-red-600 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-opacity-50 transition duration-300 shadow-sm"
                                        data-id="<?php echo htmlspecialchars($student['id']); ?>">
                                        Hapus
                                    </button>
                                    <?php if (!empty($student['last_logged_in_device_id'])): ?>
                                        <button class="logout-student-btn bg-yellow-500 text-white py-1 px-3 rounded-md hover:bg-yellow-600 focus:outline-none focus:ring-2 focus:ring-yellow-500 focus:ring-opacity-50 transition duration-300 shadow-sm mt-2 md:mt-0"
                                            data-id="<?php echo htmlspecialchars($student['id']); ?>">
                                            Hapus Sesi Login
                                        </button>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-center text-sm font-medium">
                                    <button class="download-qr-btn bg-purple-500 text-white py-1 px-3 rounded-md hover:bg-purple-600 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-opacity-50 transition duration-300 shadow-sm"
                                        data-qr-content="<?php echo htmlspecialchars($student['qr_code_content']); ?>"
                                        data-student-id="<?php echo htmlspecialchars($student['id']); ?>">
                                        Download QR
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <div class="flex flex-wrap justify-between items-center mt-6">
                <div class="text-sm text-gray-700 dark:text-gray-300 mb-2 sm:mb-0">
                    Menampilkan <?php echo min($records_per_page, count($students)); ?> dari <?php echo $total_records; ?> siswa
                </div>
                <div class="flex items-center space-x-2">
                    <?php
                    // Parameter URL dasar untuk paginasi
                    $base_url_params = [
                        'search' => $search_term,
                        'kelas_filter' => $filter_class
                    ];
                    ?>
                    <?php if ($current_page > 1): ?>
                        <a href="?<?php echo http_build_query(array_merge($base_url_params, ['page' => $current_page - 1])); ?>" class="bg-gray-300 text-gray-800 py-2 px-4 rounded-lg hover:bg-gray-400 transition duration-300 shadow-sm dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600">
                            Sebelumnya
                        </a>
                    <?php endif; ?>

                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <a href="?<?php echo http_build_query(array_merge($base_url_params, ['page' => $i])); ?>" class="py-2 px-4 rounded-lg transition duration-300 shadow-sm <?php echo ($i == $current_page) ? 'bg-indigo-600 text-white' : 'bg-gray-200 text-gray-700 hover:bg-gray-300 dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600'; ?>">
                            <?php echo $i; ?>
                        </a>
                    <?php endfor; ?>

                    <?php if ($current_page < $total_pages): ?>
                        <a href="?<?php echo http_build_query(array_merge($base_url_params, ['page' => $current_page + 1])); ?>" class="bg-gray-300 text-gray-800 py-2 px-4 rounded-lg hover:bg-gray-400 transition duration-300 shadow-sm dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600">
                            Berikutnya
                        </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Add/Edit Student Modal -->
        <div id="studentModal" class="fixed inset-0 flex items-center justify-center modal hidden dark:bg-black dark:bg-opacity-70">
            <div class="bg-white rounded-lg p-6 sm:p-8 shadow-xl w-full max-w-lg modal-content relative mx-4 dark:bg-gray-800 dark:text-gray-200">
                <button id="closeModalBtn" class="absolute top-3 right-3 text-gray-400 hover:text-gray-600 text-2xl font-bold dark:text-gray-400 dark:hover:text-gray-300">&times;</button>
                <h2 id="modalTitle" class="text-2xl font-bold text-gray-800 mb-6 text-center dark:text-gray-100">Tambah Siswa Baru</h2>
                <form id="studentForm" method="POST" action="daftarsiswa.php">
                    <input type="hidden" name="form_action" id="formAction">
                    <input type="hidden" name="original_id" id="originalId">

                    <div class="mb-4" id="idFieldContainer">
                        <label for="id" class="block text-gray-700 text-sm font-bold mb-2 dark:text-gray-300">ID Siswa:</label>
                        <input type="text" id="id" name="id" class="shadow-sm appearance-none border rounded-lg w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:ring-2 focus:ring-indigo-400 focus:border-transparent dark:bg-gray-900 dark:text-gray-200 dark:border-gray-600" readonly>
                        <p class="text-xs text-gray-500 mt-1 dark:text-gray-400">ID Siswa akan dibuat otomatis saat menambah.</p>
                    </div>
                    <div class="mb-4" id="qrContentFieldContainer">
                        <label for="qr_code_content" class="block text-gray-700 text-sm font-bold mb-2 dark:text-gray-300">Konten QR:</label>
                        <input type="text" id="qr_code_content" name="qr_code_content" class="shadow-sm appearance-none border rounded-lg w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:ring-2 focus:ring-indigo-400 focus:border-transparent dark:bg-gray-900 dark:text-gray-200 dark:border-gray-600" readonly>
                        <p class="text-xs text-gray-500 mt-1 dark:text-gray-400" id="qrContentNote">Konten QR akan otomatis sama dengan ID siswa saat menambah.</p>
                    </div>
                    <div class="mb-4">
                        <label for="nama" class="block text-gray-700 text-sm font-bold mb-2 dark:text-gray-300">Nama:</label>
                        <input type="text" id="nama" name="nama" class="shadow-sm appearance-none border rounded-lg w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:ring-2 focus:ring-indigo-400 focus:border-transparent dark:bg-gray-900 dark:text-gray-200 dark:border-gray-600" required>
                    </div>
                    <div class="mb-4">
                        <label for="kelas" class="block text-gray-700 text-sm font-bold mb-2 dark:text-gray-300">Kelas:</label>
                        <select id="kelas" name="kelas" class="shadow-sm appearance-none border rounded-lg w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:ring-2 focus:ring-indigo-400 focus:border-transparent dark:bg-gray-900 dark:text-gray-200 dark:border-gray-600" required>
                            <option value="">Pilih Kelas</option>
                            <?php foreach ($available_classes as $class_option): ?>
                                <option value="<?php echo htmlspecialchars($class_option); ?>">
                                    <?php echo htmlspecialchars($class_option); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-6">
                        <label for="nomor_wa" class="block text-gray-700 text-sm font-bold mb-2 dark:text-gray-300">Nomor WA:</label>
                        <input type="text" id="nomor_wa" name="nomor_wa" class="shadow-sm appearance-none border rounded-lg w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:ring-2 focus:ring-indigo-400 focus:border-transparent dark:bg-gray-900 dark:text-gray-200 dark:border-gray-600">
                    </div>
                    <div class="flex items-center justify-end">
                        <button type="submit" id="submitBtn" class="bg-indigo-600 text-white font-semibold py-2 px-4 rounded-lg hover:bg-indigo-700 focus:outline-none focus:ring-4 focus:ring-indigo-500 focus:ring-opacity-50 transition duration-300 shadow-md">
                            Simpan
                        </button>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- Import Excel Modal -->
        <div id="importModal" class="fixed inset-0 flex items-center justify-center modal hidden dark:bg-black dark:bg-opacity-70">
            <div class="bg-white rounded-lg p-6 sm:p-8 shadow-xl w-full max-w-lg modal-content relative mx-4 dark:bg-gray-800 dark:text-gray-200">
                <button id="closeImportModalBtn" class="absolute top-3 right-3 text-gray-400 hover:text-gray-600 text-2xl font-bold dark:text-gray-400 dark:hover:text-gray-300">&times;</button>
                <h2 class="text-2xl font-bold text-gray-800 mb-6 text-center dark:text-gray-100">Impor Data Siswa dari Excel</h2>
                <form id="importExcelFormModal" action="daftarsiswa.php" method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="import_excel" value="1">
                    <div class="mb-4">
                        <label for="excelFileInputModal" class="block text-gray-700 text-sm font-bold mb-2 dark:text-gray-300">Pilih File Excel:</label>
                        <input type="file" id="excelFileInputModal" name="excel_file" accept=".xls,.xlsx" required class="block w-full text-sm text-gray-500
                            file:mr-4 file:py-2 file:px-4
                            file:rounded-full file:border-0
                            file:text-sm file:font-semibold
                            file:bg-indigo-50 file:text-indigo-700
                            hover:file:bg-indigo-100
                            cursor-pointer dark:file:bg-indigo-700 dark:file:text-indigo-100">
                        <p class="text-xs text-gray-500 mt-1 dark:text-gray-400">Unggah file .xls atau .xlsx. Pastikan format kolom 'Nama', 'Kelas', dan 'Nomor WA' sesuai.</p>
                    </div>
                    <div class="flex items-center justify-end gap-3">
                        <button type="submit" class="bg-green-600 text-white font-semibold py-2 px-4 rounded-lg hover:bg-green-700 focus:outline-none focus:ring-4 focus:ring-green-500 focus:ring-opacity-50 transition duration-300 shadow-md">
                            <svg class="w-5 h-5 mr-2 inline-block" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4"></path></svg>
                            Mulai Impor
                        </button>
                        <a href="daftarsiswa.php?action=download_template" target="_blank" class="bg-blue-600 text-white font-semibold py-2 px-4 rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-4 focus:ring-blue-500 focus:ring-opacity-50 transition duration-300 shadow-md flex items-center justify-center">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 13l-7 7m0 0l-7-7m7 7V4"></path></svg>
                            Download Template
                        </a>
                    </div>
                </form>
            </div>
        </div>

        <!-- Canvas for QR Code Generation (hidden) -->
        <canvas id="qrDownloadCanvas"></canvas>

        <!-- Footer Powered By -->
        <div class="text-center mt-10">
            <img src="digipas.png" alt="Logo Digipas.id" class="mx-auto mb-2 w-16 h-auto" onerror="this.onerror=null;this.src='https://placehold.co/64x16/AAAAAA/FFFFFF?text=Logo+Error';">
            <p class="text-sm text-gray-600 dark:text-gray-400">
                Powered by: digipas.id &copy; <span id="currentYear"></span>
            </p>
        </div>
    </main>

    <script>
        // --- Theme Toggle Logic ---
        const themeToggle = document.getElementById('themeToggle');
        const htmlElement = document.documentElement;
        const sunIcon = document.querySelector('.sun-icon');
        const moonIcon = document.querySelector('.moon-icon');
        
        function applyTheme(theme) {
            if (theme === 'dark') {
                htmlElement.classList.add('dark');
                sunIcon.classList.add('hidden');
                moonIcon.classList.remove('hidden');
            } else {
                htmlElement.classList.remove('dark');
                sunIcon.classList.remove('hidden');
                moonIcon.classList.add('hidden'); 
            }
        }

        const savedTheme = localStorage.getItem('theme');
        if (savedTheme) {
            applyTheme(savedTheme);
        } else if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            applyTheme('dark');
        } else {
            applyTheme('light');
        }

        themeToggle.addEventListener('click', () => {
            if (htmlElement.classList.contains('dark')) {
                applyTheme('light');
                localStorage.setItem('theme', 'light');
            } else {
                localStorage.setItem('theme', 'dark');
                applyTheme('dark');
            }
        });

        document.getElementById('currentYear').textContent = new Date().getFullYear();

        // --- Sidebar Toggle Logic ---
        const manajemenPenggunaDropdownToggle = document.getElementById('manajemenPenggunaDropdownToggle');
        const manajemenPenggunaSubmenu = document.getElementById('manajemenPenggunaSubmenu');
        const dropdownArrow = document.getElementById('dropdownArrow');

        if (manajemenPenggunaDropdownToggle && manajemenPenggunaSubmenu && dropdownArrow) {
            manajemenPenggunaDropdownToggle.addEventListener('click', (e) => {
                e.preventDefault();
                manajemenPenggunaSubmenu.classList.toggle('hidden');
                dropdownArrow.classList.toggle('rotate-180');
            });
        }

        const sidebarToggle = document.getElementById('sidebarToggle');
        const sidebar = document.getElementById('sidebar');
        const sidebarToggleIcon = document.querySelector('.sidebar-toggle-icon');
        
        function initializeSidebar() {
            const isMinimized = localStorage.getItem('sidebarMinimized') === 'true';
            if (isMinimized) {
                sidebar.classList.add('minimized');
                sidebarToggleIcon.setAttribute('d', 'M9 5l7 7-7 7'); // Arrow pointing right
            } else {
                sidebar.classList.remove('minimized');
                sidebarToggleIcon.setAttribute('d', 'M4 6h16M4 12h16M4 18h16'); // Hamburger icon
            }
            // Ensure event listener is only added once
            if (!sidebarToggle.dataset.listenerAdded) {
                sidebarToggle.addEventListener('click', toggleSidebar);
                sidebarToggle.dataset.listenerAdded = true;
            }
        }

        function toggleSidebar() {
            const isCurrentlyMinimized = sidebar.classList.contains('minimized');
            if (isCurrentlyMinimized) {
                sidebar.classList.remove('minimized');
                sidebarToggleIcon.setAttribute('d', 'M4 6h16M4 12h16M4 18h16');
                localStorage.setItem('sidebarMinimized', 'true');
            } else {
                sidebar.classList.add('minimized');
                sidebarToggleIcon.setAttribute('d', 'M9 5l7 7-7 7');
                localStorage.setItem('sidebarMinimized', 'false');
            }
        }
        
        // Initialize sidebar on DOMContentLoaded (page load)
        document.addEventListener('DOMContentLoaded', initializeSidebar);


        // --- Modal & Form Handling Logic ---
        const studentModal = document.getElementById('studentModal');
        const openAddModalBtn = document.getElementById('openAddModalBtn');
        const closeModalBtn = document.getElementById('closeModalBtn');
        const modalTitle = document.getElementById('modalTitle');
        const studentForm = document.getElementById('studentForm');
        const formAction = document.getElementById('formAction');
        const originalId = document.getElementById('originalId');
        const idInput = document.getElementById('id');
        const qrCodeContentInput = document.getElementById('qr_code_content');
        const namaInput = document.getElementById('nama');
        const kelasInput = document.getElementById('kelas');
        const nomorWaInput = document.getElementById('nomor_wa');
        const submitBtn = document.getElementById('submitBtn');
        const qrContentNote = document.getElementById('qrContentNote');
        
        const idFieldContainer = document.getElementById('idFieldContainer');
        const qrContentFieldContainer = document.getElementById('qrContentFieldContainer');

        const importModal = document.getElementById('importModal');
        const openImportModalBtn = document.getElementById('openImportModalBtn');
        const closeImportModalBtn = document.getElementById('closeImportModalBtn');
        const excelFileInputModal = document.getElementById('excelFileInputModal');
        const importExcelFormModal = document.getElementById('importExcelFormModal');


        // Function to open Add Student modal
        openAddModalBtn.addEventListener('click', () => {
            modalTitle.textContent = 'Tambah Siswa Baru';
            formAction.name = 'add_student'; // Set form action for adding

            // Hide ID and QR content fields for new student (auto-generated)
            idFieldContainer.classList.add('hidden');
            qrContentFieldContainer.classList.add('hidden');
            qrCodeContentInput.removeAttribute('required'); // QR content not required for add

            // Clear form fields
            idInput.value = '';
            qrCodeContentInput.value = '';
            namaInput.value = '';
            kelasInput.value = '';
            nomorWaInput.value = '';
            studentModal.classList.remove('hidden'); // Show modal
        });

        // Event listener for Edit buttons
        document.querySelectorAll('.edit-btn').forEach(button => {
            button.addEventListener('click', (event) => {
                const data = event.target.dataset; // Get data attributes
                modalTitle.textContent = 'Edit Data Siswa';
                formAction.name = 'edit_student'; // Set form action for editing
                originalId.value = data.id; // Store original ID

                // Show ID and QR content fields for editing
                idFieldContainer.classList.remove('hidden');
                idInput.value = data.id;
                idInput.readOnly = true; // ID cannot be changed

                qrContentFieldContainer.classList.remove('hidden');
                qrCodeContentInput.setAttribute('required', 'required'); // QR content required for edit
                qrCodeContentInput.readOnly = false; // QR content can be edited

                // Populate form fields with existing data
                qrCodeContentInput.value = data.qr;
                namaInput.value = data.nama;
                kelasInput.value = data.kelas;
                nomorWaInput.value = data.wa;
                studentModal.classList.remove('hidden'); // Show modal
            });
        });

        // Close modal button
        closeModalBtn.addEventListener('click', () => {
            studentModal.classList.add('hidden'); // Hide modal
            // Reset visibility/readonly state for next open
            idFieldContainer.classList.remove('hidden');
            qrContentFieldContainer.classList.remove('hidden');
            idInput.readOnly = false;
            qrCodeContentInput.readOnly = false;
            qrCodeContentInput.setAttribute('required', 'required');
        });

        // Close modal when clicking outside
        studentModal.addEventListener('click', (event) => {
            if (event.target === studentModal) {
                studentModal.classList.add('hidden');
                // Reset visibility/readonly state for next open
                idFieldContainer.classList.remove('hidden');
                qrContentFieldContainer.classList.remove('hidden');
                idInput.readOnly = false;
                qrCodeContentInput.readOnly = false;
                qrCodeContentInput.setAttribute('required', 'required');
            }
        });

        // Open Import Excel modal
        openImportModalBtn.addEventListener('click', () => {
            importModal.classList.remove('hidden');
        });

        // Close Import Excel modal button
        closeImportModalBtn.addEventListener('click', () => {
            importModal.classList.add('hidden');
            excelFileInputModal.value = ''; // Clear selected file
        });

        // Close Import Excel modal when clicking outside
        importModal.addEventListener('click', (event) => {
            if (event.target === importModal) {
                importModal.classList.add('hidden');
                excelFileInputModal.value = ''; // Clear selected file
            }
        });

        // Event listener for Delete buttons (NO MODAL CONFIRMATION)
        document.querySelectorAll('.delete-btn').forEach(button => { 
            button.addEventListener('click', (event) => { 
                const studentId = event.target.dataset.id; 
                console.log('Attempting to delete student with ID (direct action):', studentId); // Debugging line
                // Directly redirect to delete action without confirmation modal
                window.location.href = `daftarsiswa.php?action=delete&id=${studentId}&${new URLSearchParams(window.location.search).toString()}`; 
            }); 
        }); 

        // Event listener for "Hapus Sesi Login" button (NO MODAL CONFIRMATION)
        document.querySelectorAll('.logout-student-btn').forEach(button => {
            button.addEventListener('click', (event) => {
                const studentId = event.target.dataset.id; // Get student ID directly from data-id attribute
                console.log('Attempting to logout student with ID (direct action):', studentId); // Debugging line
                // Directly redirect to logout action without confirmation modal
                window.location.href = `daftarsiswa.php?action=logout_student&id=${studentId}&${new URLSearchParams(window.location.search).toString()}`;
            });
        });

        // Removed createCustomConfirm function as it's no longer used for delete actions

        // --- Message Alert Handling ---
        window.addEventListener('load', () => {
            const urlParams = new URLSearchParams(window.location.search);
            const msg = urlParams.get('message');
            const type = urlParams.get('type');

            if (msg && type) {
                const messageDiv = document.getElementById('messageAlert');
                if (messageDiv) {
                    messageDiv.classList.remove('hidden');
                    messageDiv.querySelector('strong').textContent = type === 'success' ? 'Sukses!' : 'Error!'; // Update strong tag too
                    messageDiv.querySelector('span').textContent = decodeURIComponent(msg);
                    if (type === 'success') {
                        messageDiv.classList.add('bg-green-100', 'border-green-400', 'text-green-700');
                        messageDiv.classList.remove('bg-red-100', 'border-red-400', 'text-red-700');
                    } else {
                        messageDiv.classList.add('bg-red-100', 'border-red-400', 'text-red-700');
                        messageDiv.classList.remove('bg-green-100', 'border-green-400', 'text-green-700');
                    }

                    // Hide message after 3 seconds
                    setTimeout(() => {
                        messageDiv.classList.add('hidden');
                    }, 3000);

                    // Clean URL by removing message parameters
                    const newUrl = window.location.origin + window.location.pathname;
                    const paramsToKeep = new URLSearchParams(window.location.search);
                    paramsToKeep.delete('message');
                    paramsToKeep.delete('type');
                    window.history.replaceState({}, document.title, `${newUrl}?${paramsToKeep.toString()}`);
                }
            }
        });

        // --- QR Code Download Logic ---
        const qrDownloadCanvas = document.getElementById('qrDownloadCanvas');

        function downloadQrCode(qrContent, studentId) {
            qrDownloadCanvas.width = 250;
            qrDownloadCanvas.height = 250;

            const qr = new QRious({
                element: qrDownloadCanvas,
                value: qrContent,
                size: 250
            });

            const link = document.createElement('a');
            link.download = `QR_Siswa_${studentId}.png`;
            link.href = qrDownloadCanvas.toDataURL('image/png');
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }

        document.querySelectorAll('.download-qr-btn').forEach(button => {
            button.addEventListener('click', (event) => {
                const qrContent = event.target.dataset.qrContent;
                const studentId = event.target.dataset.studentId;
                downloadQrCode(qrContent, studentId);
            });
        });

        // --- Search and Filter Logic ---
        const searchInput = document.getElementById('searchInput');
        const classFilterDropdown = document.getElementById('classFilterDropdown');

        /**
         * Builds the URL with current search and filter parameters.
         * @param {number} page The target page number.
         * @returns {string} The constructed URL.
         */
        function buildUrl(page = 1) {
            const newSearchParams = new URLSearchParams();

            const searchTerm = searchInput.value;
            const filterClass = classFilterDropdown.value;

            if (searchTerm) {
                newSearchParams.append('search', searchTerm);
            }
            if (filterClass !== 'all') {
                newSearchParams.append('kelas_filter', filterClass);
            }
            newSearchParams.append('page', page);

            return `daftarsiswa.php?${newSearchParams.toString()}`;
        }

        // Apply filters and search when input changes or dropdown changes
        function applyFiltersAndSearch() {
            window.location.href = buildUrl(1); // Reset to page 1 on new search/filter
        }

        let searchTimeout;
        searchInput.addEventListener('keyup', () => {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(() => {
                applyFiltersAndSearch();
            }, 500); // Debounce search input
        });

        classFilterDropdown.addEventListener('change', () => {
            applyFiltersAndSearch();
        });

    </script>
</body>
</html>
