<?php
// Aktifkan pelaporan kesalahan PHP untuk debugging. Hapus atau atur ke 0 di lingkungan produksi.
error_reporting(E_ALL); // Laporkan semua jenis kesalahan
ini_set('display_errors', 1); // Tampilkan kesalahan di layar

// Mulai sesi PHP
session_start();

// Sertakan file konfigurasi database
// Pastikan path ini benar relatif terhadap lokasi file ini (misal: jika pengawas.php di admin/, maka config.php ada di folder induk).
require_once '../config.php';

// Inisialisasi variabel pesan
$message = '';
$message_type = ''; // 'success' atau 'error'

// Direktori upload dasar (relatif terhadap lokasi file pengawas.php)
// Ini berarti folder 'uploads' diharapkan berada di direktori yang sama dengan 'pengawas.php'.
$baseUploadDir = 'uploads/';

// Direktori spesifik untuk foto profil pengawas
$profilePicsDir = $baseUploadDir . 'profile_pics/';
if (!is_dir($profilePicsDir)) {
    // Buat direktori jika belum ada. Pastikan izin tulis (0777 untuk development, sesuaikan di produksi).
    mkdir($profilePicsDir, 0777, true);
}

// Periksa apakah pengguna sudah login
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== TRUE) {
    header("location: index.php");
    exit;
}

// Periksa peran pengguna. Hanya admin yang bisa mengakses halaman ini.
if ($_SESSION['role'] !== 'admin') {
    header("location: index.php");
    exit;
}

// Data pengguna yang akan ditampilkan di halaman
$nama_pengguna = htmlspecialchars($_SESSION['nama']);

// Fungsi untuk mendapatkan pengaturan sekolah
function getSchoolSettings($conn) {
    $sql = "SELECT nama_sekolah, alamat_sekolah, logo_sekolah FROM pengaturan_sekolah WHERE id = 1 LIMIT 1";
    $result = $conn->query($sql);
    if ($result && $result->num_rows > 0) {
        return $result->fetch_assoc();
    }
    return ['nama_sekolah' => 'Nama Sekolah Default', 'alamat_sekolah' => '', 'logo_sekolah' => ''];
}

// Fungsi untuk mendapatkan daftar pengawas
function getSupervisors($conn, $searchTerm = '') {
    $supervisors = [];
    $sql = "SELECT id, username, email, nama, wa, foto, tanggal_pembuatan FROM pengguna WHERE role = 'pengawas'";
    
    // Tambahkan pencarian jika ada searchTerm
    if (!empty($searchTerm)) {
        $searchTerm = '%' . $conn->real_escape_string($searchTerm) . '%';
        $sql .= " AND (username LIKE ? OR nama LIKE ? OR email LIKE ? OR wa LIKE ?)";
        $stmt = $conn->prepare($sql);
        // Periksa apakah prepare berhasil sebelum bind dan execute
        if ($stmt) {
            $stmt->bind_param("ssss", $searchTerm, $searchTerm, $searchTerm, $searchTerm);
            $stmt->execute();
            $result = $stmt->get_result();
        } else {
            error_log("Failed to prepare statement for getSupervisors: " . $conn->error);
            return []; // Kembalikan array kosong jika prepare gagal
        }
    } else {
        $result = $conn->query($sql);
    }

    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $supervisors[] = $row;
        }
    }
    return $supervisors;
}

// Fungsi untuk mendapatkan detail pengawas berdasarkan ID
function getSupervisorById($conn, $id) {
    $stmt = $conn->prepare("SELECT id, username, email, nama, wa, foto FROM pengguna WHERE id = ? AND role = 'pengawas'");
    // Periksa apakah prepare berhasil
    if ($stmt) {
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    } else {
        error_log("Failed to prepare statement for getSupervisorById: " . $conn->error);
        return null;
    }
}

// Fungsi untuk menambah pengawas baru
function addSupervisor($conn, $username, $email, $kata_sandi, $nama, $wa, $foto, $profilePicsDir) {
    $hashed_password = password_hash($kata_sandi, PASSWORD_DEFAULT);

    // Cek apakah username atau email sudah ada
    $check_stmt = $conn->prepare("SELECT COUNT(*) FROM pengguna WHERE username = ? OR email = ?");
    if ($check_stmt) {
        $check_stmt->bind_param("ss", $username, $email);
        $check_stmt->execute();
        $check_stmt->bind_result($count);
        $check_stmt->fetch();
        $check_stmt->close();
    } else {
        return ['status' => false, 'message' => 'Gagal mempersiapkan statement untuk pengecekan duplikasi: ' . $conn->error];
    }

    if ($count > 0) {
        return ['status' => false, 'message' => 'Username atau Email sudah terdaftar.'];
    }

    $stmt = $conn->prepare("INSERT INTO pengguna (username, email, kata_sandi, nama, wa, foto, role) VALUES (?, ?, ?, ?, ?, ?, 'pengawas')");
    if ($stmt) {
        $stmt->bind_param("ssssss", $username, $email, $hashed_password, $nama, $wa, $foto);
        
        if ($stmt->execute()) {
            return ['status' => true, 'message' => 'Pengawas berhasil ditambahkan.'];
        } else {
            return ['status' => false, 'message' => 'Gagal menambahkan pengawas: ' . $stmt->error];
        }
    } else {
        return ['status' => false, 'message' => 'Gagal mempersiapkan statement penambahan: ' . $conn->error];
    }
}

// Fungsi untuk memperbarui pengawas
function updateSupervisor($conn, $id, $username, $email, $kata_sandi, $nama, $wa, $foto, $profilePicsDir) {
    $sql = "UPDATE pengguna SET username = ?, email = ?, nama = ?, wa = ?, foto = ? ";
    $params = [$username, $email, $nama, $wa, $foto];
    $types = "sssss";

    if (!empty($kata_sandi)) {
        $hashed_password = password_hash($kata_sandi, PASSWORD_DEFAULT);
        $sql .= ", kata_sandi = ? ";
        $params[] = $hashed_password;
        $types .= "s";
    }
    $sql .= " WHERE id = ? AND role = 'pengawas'";
    $params[] = $id;
    $types .= "i";

    $stmt = $conn->prepare($sql);
    if ($stmt) {
        // Menggunakan call_user_func_array untuk bind_param
        // Referensi: https://www.php.net/manual/en/mysqli-stmt.bind-param.php#example-mysqli-stmt.bind-param-simple
        $bind_names[] = $types;
        for ($i = 0; $i < count($params); $i++) {
            $bind_name = 'bind' . $i;
            $$bind_name = &$params[$i];
            $bind_names[] = &$$bind_name;
        }
        call_user_func_array([$stmt, 'bind_param'], $bind_names);

        if ($stmt->execute()) {
            return ['status' => true, 'message' => 'Pengawas berhasil diperbarui.'];
        } else {
            return ['status' => false, 'message' => 'Gagal memperbarui pengawas: ' . $stmt->error];
        }
    } else {
        return ['status' => false, 'message' => 'Gagal mempersiapkan statement pembaruan: ' . $conn->error];
    }
}

// Fungsi untuk menghapus pengawas
function deleteSupervisor($conn, $id, $profilePicsDir) {
    // Ambil nama foto sebelum menghapus record
    $supervisor = getSupervisorById($conn, $id);
    if ($supervisor && !empty($supervisor['foto']) && file_exists($profilePicsDir . $supervisor['foto'])) {
        unlink($profilePicsDir . $supervisor['foto']); // Hapus file foto dari server
    }

    $stmt = $conn->prepare("DELETE FROM pengguna WHERE id = ? AND role = 'pengawas'");
    if ($stmt) {
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            return ['status' => true, 'message' => 'Pengawas berhasil dihapus.'];
        } else {
            return ['status' => false, 'message' => 'Gagal menghapus pengawas: ' . $stmt->error];
        }
    } else {
        return ['status' => false, 'message' => 'Gagal mempersiapkan statement penghapusan: ' . $conn->error];
    }
}

// Handle AJAX search request
if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
    $searchTerm = $_GET['search'] ?? '';
    $supervisors = getSupervisors($conn, $searchTerm);
    echo renderSupervisorsTableRows($supervisors, $profilePicsDir); // Gunakan $profilePicsDir
    exit; // Penting: Hentikan eksekusi setelah mengirimkan respon AJAX
}

// Handle form submissions (Add, Edit, Delete)
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];

        // Common fields for Add and Edit
        $id = $_POST['id'] ?? null;
        $username = trim($_POST['username'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $kata_sandi = $_POST['kata_sandi'] ?? ''; // Hanya jika ingin mengubah password
        $nama = trim($_POST['nama'] ?? '');
        $wa = trim($_POST['wa'] ?? '');
        $current_foto = $_POST['current_foto'] ?? ''; // Foto lama saat edit

        $foto = $current_foto; // Default: gunakan foto lama

        // Handle foto upload
        if (isset($_FILES['foto']) && $_FILES['foto']['error'] == UPLOAD_ERR_OK) {
            $fileTmpPath = $_FILES['foto']['tmp_name'];
            $fileName = $_FILES['foto']['name'];
            $fileSize = $_FILES['foto']['size'];
            $fileType = $_FILES['foto']['type'];
            $fileNameCmps = explode(".", $fileName);
            $fileExtension = strtolower(end($fileNameCmps));

            $newFileName = md5(time() . $fileName) . '.' . $fileExtension;
            $destPath = $profilePicsDir . $newFileName; // Gunakan $profilePicsDir

            $allowedFileTypes = ['jpg', 'jpeg', 'png', 'gif'];
            if (in_array($fileExtension, $allowedFileTypes)) {
                if ($fileSize < 5000000) { // Max 5MB
                    if (move_uploaded_file($fileTmpPath, $destPath)) {
                        $foto = $newFileName; // Simpan nama file baru
                        // Hapus foto lama jika ada dan berbeda dengan yang baru
                        if (!empty($current_foto) && $current_foto != $newFileName && file_exists($profilePicsDir . $current_foto)) { // Gunakan $profilePicsDir
                            unlink($profilePicsDir . $current_foto); // Gunakan $profilePicsDir
                        }
                    } else {
                        $message = "Gagal mengunggah foto.";
                        $message_type = "error";
                    }
                } else {
                    $message = "Ukuran foto terlalu besar (maks 5MB).";
                    $message_type = "error";
                }
            } else {
                $message = "Format foto tidak didukung (hanya JPG, JPEG, PNG, GIF).";
                $message_type = "error";
            }
        } else if (isset($_POST['remove_foto']) && $_POST['remove_foto'] == '1') {
            // Hapus foto jika checkbox "hapus foto" dicentang
            if (!empty($current_foto) && file_exists($profilePicsDir . $current_foto)) { // Gunakan $profilePicsDir
                unlink($profilePicsDir . $current_foto); // Gunakan $profilePicsDir
            }
            $foto = ''; // Set foto menjadi kosong di database
        }


        if ($action == 'add') {
            if (empty($username) || empty($email) || empty($kata_sandi) || empty($nama) || empty($wa)) {
                $message = "Semua bidang wajib diisi, kecuali foto.";
                $message_type = "error";
            } else {
                $result = addSupervisor($conn, $username, $email, $kata_sandi, $nama, $wa, $foto, $profilePicsDir); // Gunakan $profilePicsDir
                $message = $result['message'];
                $message_type = $result['status'] ? 'success' : 'error';
            }
        } elseif ($action == 'edit') {
            if (empty($id) || empty($username) || empty($email) || empty($nama) || empty($wa)) {
                $message = "ID dan semua bidang wajib diisi, kecuali password dan foto.";
                $message_type = "error";
            } else {
                $result = updateSupervisor($conn, $id, $username, $email, $kata_sandi, $nama, $wa, $foto, $profilePicsDir); // Gunakan $profilePicsDir
                $message = $result['message'];
                $message_type = $result['status'] ? 'success' : 'error';
            }
        } elseif ($action == 'delete') {
            $id_to_delete = $_POST['id_to_delete'] ?? null;
            if (!empty($id_to_delete)) {
                $result = deleteSupervisor($conn, $id_to_delete, $profilePicsDir); // Gunakan $profilePicsDir
                $message = $result['message'];
                $message_type = $result['status'] ? 'success' : 'error';
            } else {
                $message = "ID pengawas tidak valid untuk dihapus.";
                $message_type = "error";
            }
        }
    }
}

// Ambil pengaturan sekolah
$settings = getSchoolSettings($conn);
$nama_sekolah_display = htmlspecialchars($settings['nama_sekolah']);
// Perbaikan path logo sekolah agar selalu relatif terhadap baseUploadDir
$logo_sekolah_path = !empty($settings['logo_sekolah']) ? $baseUploadDir . htmlspecialchars($settings['logo_sekolah']) : 'https://placehold.co/60x60/B0E0E6/000000?text=LOGO';

// Ambil daftar pengawas untuk ditampilkan
$supervisors = getSupervisors($conn);

// Fungsi untuk merender baris tabel pengawas (digunakan oleh HTML dan AJAX)
function renderSupervisorsTableRows($supervisors, $profilePicsDir) { // Parameter diubah menjadi $profilePicsDir
    ob_start(); // Mulai output buffering
    if (empty($supervisors)) {
        echo '<tr><td colspan="8" class="py-4 text-center text-gray-500 dark:text-gray-400">Tidak ada data pengawas.</td></tr>';
    } else {
        foreach ($supervisors as $supervisor) {
            $foto_path = !empty($supervisor['foto']) ? $profilePicsDir . htmlspecialchars($supervisor['foto']) : 'https://placehold.co/50x50/cccccc/000000?text=NO+IMG';
            echo '<tr class="border-b dark:border-gray-700">';
            echo '<td class="py-3 px-4 text-center">' . htmlspecialchars($supervisor['id']) . '</td>';
            echo '<td class="py-3 px-4">' . htmlspecialchars($supervisor['username']) . '</td>';
            echo '<td class="py-3 px-4">' . htmlspecialchars($supervisor['email']) . '</td>';
            echo '<td class="py-3 px-4">' . htmlspecialchars($supervisor['nama']) . '</td>';
            echo '<td class="py-3 px-4">' . htmlspecialchars($supervisor['wa']) . '</td>';
            echo '<td class="py-3 px-4 text-center"><img src="' . $foto_path . '" alt="Foto Pengawas" class="w-12 h-12 object-cover rounded-full mx-auto" onerror="this.onerror=null;this.src=\'https://placehold.co/50x50/cccccc/000000?text=NO+IMG\';"></td>';
            echo '<td class="py-3 px-4 text-center">' . date('d-m-Y H:i', strtotime($supervisor['tanggal_pembuatan'])) . '</td>';
            echo '<td class="py-3 px-4 text-center">';
            echo '<button onclick="openEditModal(' . htmlspecialchars(json_encode($supervisor)) . ')" class="bg-blue-500 hover:bg-blue-600 text-white font-bold py-1 px-3 rounded-md text-sm mr-2 mb-1">Edit</button>';
            echo '<button onclick="confirmDelete(' . $supervisor['id'] . ')" class="bg-red-500 hover:bg-red-600 text-white font-bold py-1 px-3 rounded-md text-sm mb-1">Hapus</button>';
            echo '</td>';
            echo '</tr>';
        }
    }
    return ob_get_clean(); // Kembalikan konten buffering
}

$conn->close(); // Tutup koneksi database setelah semua operasi
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manajemen Pengawas - <?php echo $nama_pengguna; ?></title>
    <!-- Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Font Inter dari Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            transition: background-color 0.5s ease-in-out, color 0.5s ease-in-out;
            overflow-x: hidden; /* Mencegah scroll horizontal */
        }
        /* Mode gelap: Latar belakang yang lebih dalam */
        html.dark body {
            background-color: #0b0c10;
            color: #e2e8f0;
        }
        /* Mode terang: Latar belakang yang lebih hidup */
        html:not(.dark) body {
            background-color: #e0e7ff;
            color: #1a202c;
        }
        /* Transisi untuk elemen di dalam kartu */
        .glass-effect {
            transition: background-color 0.5s ease-in-out, border-color 0.5s ease-in-out, box-shadow 0.5s ease-in-out;
            background-color: rgba(255, 255, 255, 0.8); /* Latar belakang terang */
            backdrop-filter: blur(10px); /* Efek blur */
            border: 1px solid rgba(255, 255, 255, 0.3); /* Border samar */
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        html.dark .glass-effect {
            background-color: rgba(30, 41, 59, 0.8); /* Latar belakang gelap */
            border: 1px solid rgba(45, 55, 72, 0.5);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.2);
        }
        /* Mode gelap untuk teks */
        html.dark .text-day-mode-black {
            color: #e2e8f0;
        }
        /* Mode terang untuk teks: tetap hitam */
        html:not(.dark) .text-day-mode-black {
            color: #000000;
        }
        html.dark .sidebar {
            background-color: #1a202c;
            border-right-color: #2d3748;
        }
        html:not(.dark) .sidebar {
            background-color: #ffffff;
            border-right-color: #e2e8f0;
        }
        html.dark .sidebar a {
            color: #a0aec0;
        }
        html.dark .sidebar a:hover {
            background-color: #2d3748;
            color: #ffffff;
        }
        html:not(.dark) .sidebar a {
            color: #4a5568;
        }
        html:not(.dark) .sidebar a:hover {
            background-color: #edf2f7;
            color: #2d3748;
        }
        /* Gaya untuk tombol pengalih tema */
        #themeToggle {
            transition: background-color 0.5s ease-in-out, color 0.5s ease-in-out, box-shadow 0.5s ease-in-out;
        }

        /* Gaya untuk sidebar minimisasi */
        .sidebar {
            width: 256px; /* Lebar default (w-64) */
            transition: width 0.3s ease-in-out; /* Transisi untuk perubahan lebar */
            flex-shrink: 0; /* Mencegah sidebar menyusut saat konten utama lebar */
        }
        .sidebar.minimized {
            width: 64px; /* Lebar saat diminimalisir (w-16) */
        }
        .sidebar.minimized .sidebar-text {
            display: none; /* Sembunyikan teks menu */
        }
        .sidebar.minimized .sidebar-icon {
            margin-right: 0 !important; /* Hapus margin dari ikon */
            /* Mengatur ukuran ikon agar terlihat jelas saat diminimalisir */
            width: 24px; /* Sedikit lebih besar dari w-5 (20px) */
            height: 24px; /* Sedikit lebih besar dari h-5 (20px) */
        }
        .sidebar.minimized .sidebar-menu-item {
            justify-content: center; /* Pusatkan ikon */
            padding-left: 0; /* Hapus padding kiri */
            padding-right: 0; /* Hapus padding kanan */
        }
        .sidebar.minimized .sidebar-dropdown-icon {
            display: none; /* Sembunyikan ikon panah dropdown saat diminimalisir */
        }
        .sidebar.minimized .sidebar-header {
            visibility: hidden; /* Sembunyikan teks header sidebar */
            white-space: nowrap; /* Pastikan teks tidak membungkus */
            overflow: hidden; /* Sembunyikan overflow */
            width: 0; /* Atur lebar menjadi nol */
            padding-left: 0;
            padding-right: 0;
        }
        .sidebar.minimized .sidebar-toggle-button {
            justify-content: center; /* Pusatkan tombol toggle */
        }
        .sidebar.minimized .school-logo {
            margin-right: 0 !important; /* Hapus margin kanan logo */
            width: 40px; /* Sesuaikan ukuran logo saat diminimalisir */
            height: 40px; /* Sesuaikan ukuran logo saat diminimalisir */
        }
        .sidebar.minimized .school-name {
            display: none; /* Sembunyikan nama sekolah saat diminimalisir (tetap) */
        }

        /* Tooltip untuk ikon saat sidebar diminimalisir */
        .sidebar.minimized [data-tooltip]:hover:after {
            content: attr(data-tooltip); /* Ambil teks dari atribut data-tooltip */
            position: absolute;
            left: 70px; /* Sesuaikan posisi tooltip */
            top: 50%;
            transform: translateY(-50%);
            background-color: #333;
            color: #fff;
            padding: 5px 10px;
            border-radius: 4px;
            white-space: nowrap;
            z-index: 999;
            opacity: 1;
            transition: opacity 0.2s ease;
        }
        .sidebar.minimized [data-tooltip]:after {
            opacity: 0; /* Sembunyikan secara default */
            pointer-events: none; /* Jangan menghalangi interaksi mouse */
        }

        /* Gaya untuk modal */
        .modal {
            display: none; /* Hidden by default */
            position: fixed; /* Stay in place */
            z-index: 1000; /* Sit on top */
            left: 0;
            top: 0;
            width: 100%; /* Full width */
            height: 100%; /* Full height */
            overflow: auto; /* Enable scroll if needed */
            background-color: rgba(0,0,0,0.4); /* Black w/ opacity */
            align-items: center;
            justify-content: center;
        }
        .modal-content {
            margin: auto;
            padding: 20px;
            border-radius: 8px;
            width: 90%;
            max-width: 600px;
            position: relative;
        }
        .close-button {
            position: absolute;
            top: 15px;
            right: 25px;
            font-size: 30px;
            font-weight: bold;
            cursor: pointer;
        }
        .close-button:hover,
        .close-button:focus {
            color: #bbb;
            text-decoration: none;
            cursor: pointer;
        }
    </style>
</head>
<body class="flex min-h-screen">

    <!-- Sidebar -->
    <aside id="sidebar" class="sidebar w-64 p-5 border-r shadow-lg flex flex-col justify-between transition-colors duration-500 relative">
        <div>
            <div class="flex items-center justify-between mb-8">
                <div class="flex items-center sidebar-header">
                    <!-- Logo Sekolah -->
                    <img src="<?php echo $logo_sekolah_path; ?>" alt="Logo Sekolah" class="school-logo w-12 h-12 object-contain mr-3 rounded-md" onerror="this.onerror=null;this.src='https://placehold.co/60x60/B0E0E6/000000?text=LOGO';">
                    <span class="text-xl font-bold text-blue-600 dark:text-blue-400 school-name"><?php echo $nama_sekolah_display; ?></span>
                </div>
                <button id="sidebarToggle" class="p-2 rounded-full hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-toggle-button">
                    <!-- Icon untuk toggle sidebar (hamburger atau chevron) -->
                    <svg class="w-6 h-6 sidebar-toggle-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                    </svg>
                </button>
            </div>
            <nav>
                <ul class="space-y-1">
                    <li>
                        <a href="admin.php" class="flex items-center py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Dashboard">
                            <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"></path></svg>
                            <span class="sidebar-text">Dashboard</span>
                        </a>
                    </li>
                    <!-- Dropdown Menu untuk Manajemen Pengguna -->
                    <li class="relative">
                        <a href="#" id="manajemenPenggunaDropdownToggle" class="flex items-center justify-between py-2 px-3 rounded-lg bg-gray-100 dark:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Manajemen Pengguna">
                            <div class="flex items-center">
                                <!-- Icon "users" -->
                                <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h-5v-9H4L12 3l8 8h-8v9z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 11c-2.209 0-4-1.791-4-4s1.791-4 4-4 4 1.791 4 4-1.791 4-4 4zM12 13a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path></svg>
                                <span class="sidebar-text">Manajemen Pengguna</span>
                            </div>
                            <svg id="manajemenPenggunaDropdownArrow" class="w-4 h-4 sidebar-dropdown-icon transform transition-transform duration-200" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </a>
                        <ul id="manajemenPenggunaSubmenu" class="pl-8 mt-1 space-y-1">
                            <li>
                                <a href="daftarsiswa.php" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Lihat Semua Pengguna">
                                    <span class="sidebar-text">Daftar Siswa</span>
                                </a>
                            </li>
                            <li>
                                <a href="pengawas.php" class="flex items-center py-1 px-2 rounded-lg bg-gray-200 dark:bg-gray-800 transition-colors duration-200 sidebar-menu-item" data-tooltip="Manajemen Pengawas">
                                    <span class="sidebar-text">Manajemen Pengawas</span>
                                </a>
                            </li>
                             <li>
                                <a href="#" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Tambah Pengguna Baru">
                                    <span class="sidebar-text">Tambah Pengguna</span>
                                </a>
                            </li>
                        </ul>
                    </li>
                    <li>
                        <a href="#" class="flex items-center py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Laporan">
                            <!-- Icon "book" -->
                            <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.523 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5s3.332.477 4.5 1.253v13C19.832 18.523 18.246 18 16.5 18s-3.332.477-4.5 1.253"></path></svg>
                            <span class="sidebar-text">Laporan</span>
                        </a>
                    </li>
                    <!-- Dropdown Menu untuk Pengaturan -->
                    <li class="relative">
                        <a href="#" id="settingsDropdownToggle" class="flex items-center justify-between py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Pengaturan">
                            <div class="flex items-center">
                                <!-- Icon "settings" -->
                                <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.525.322 1.018.673 1.505 1.01z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path></svg>
                                <span class="sidebar-text">Pengaturan</span>
                            </div>
                            <svg id="settingsDropdownArrow" class="w-4 h-4 sidebar-dropdown-icon transform transition-transform duration-200" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </a>
                        <ul id="settingsSubmenu" class="pl-8 mt-1 space-y-1 hidden">
                            <li>
                                <a href="setting.php" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Pengaturan Umum">
                                    <span class="sidebar-text">Pengaturan Umum</span>
                                </a>
                            </li>
                            <li>
                                <a href="waapi.php" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Pengaturan WA API">
                                    <span class="sidebar-text">Pengaturan WA API</span>
                                </a>
                            </li>
                             <li>
                                <a href="token.php" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Pengaturan Token Ujian">
                                    <span class="sidebar-text">Token Ujian</span>
                                </a>
                            </li>
                        </ul>
                    </li>
                </ul>
            </nav>
        </div>
        <div class="mt-8">
            <a href="logout.php" class="flex items-center py-2 px-3 rounded-lg bg-red-500 hover:bg-red-600 text-white transition-colors duration-200 sidebar-menu-item" data-tooltip="Logout">
                <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path></svg>
                <span class="sidebar-text">Logout</span>
            </a>
        </div>
    </aside>

    <!-- Konten Utama -->
    <main id="mainContent" class="flex-1 p-10 transition-all duration-300">
        <!-- Tombol pengalih tema gelap/terang -->
        <div class="absolute top-4 right-4 z-10">
            <button id="themeToggle" class="p-2 rounded-full bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white shadow-md">
                <!-- Icon SVG untuk tema terang (matahari) -->
                <svg class="w-6 h-6 sun-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 3v1m0 16v1m9-9h1M3 12h1m15.325-7.757l-.707-.707M4.343 19.657l-.707-.707m12.728 0l-.707.707M6.343 6.343l-.707-.707"></path>
                </svg>
                <!-- Ikon bulan, tersembunyi secara default -->
                <svg class="w-6 h-6 moon-icon hidden" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9 9 0 008.354-5.646z"></path>
                </svg>
            </button>
        </div>

        <h1 class="text-4xl font-extrabold mb-8 text-day-mode-black dark:text-white">Manajemen Pengawas</h1>

        <?php if (!empty($message)): ?>
            <div class="p-4 mb-4 rounded-lg
                <?php echo $message_type === 'success' ? 'bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-100' : 'bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-100'; ?>
                shadow-md" role="alert">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div class="glass-effect p-8 rounded-lg shadow-xl mb-8">
            <div class="flex justify-between items-center mb-6">
                <h3 class="text-2xl font-semibold text-day-mode-black dark:text-white">Daftar Pengawas</h3>
                <button onclick="openAddModal()" class="px-5 py-2 bg-green-600 text-white font-semibold rounded-lg shadow-md hover:bg-green-700 transition-colors duration-200">
                    Tambah Pengawas Baru
                </button>
            </div>

            <!-- Pencarian Real-time -->
            <div class="mb-6">
                <input type="text" id="supervisorSearch" onkeyup="searchSupervisors()"
                       class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white dark:focus:ring-blue-400 dark:focus:border-blue-400"
                       placeholder="Cari pengawas (username, nama, email, WA)...">
            </div>

            <!-- Tabel Pengawas -->
            <div class="overflow-x-auto rounded-lg shadow-md">
                <table class="min-w-full bg-white dark:bg-gray-800">
                    <thead class="bg-gray-100 dark:bg-gray-700">
                        <tr>
                            <th class="py-3 px-4 text-left text-sm font-medium text-gray-700 dark:text-gray-200 uppercase tracking-wider text-center">ID</th>
                            <th class="py-3 px-4 text-left text-sm font-medium text-gray-700 dark:text-gray-200 uppercase tracking-wider">Username</th>
                            <th class="py-3 px-4 text-left text-sm font-medium text-gray-700 dark:text-gray-200 uppercase tracking-wider">Email</th>
                            <th class="py-3 px-4 text-left text-sm font-medium text-gray-700 dark:text-gray-200 uppercase tracking-wider">Nama</th>
                            <th class="py-3 px-4 text-left text-sm font-medium text-gray-700 dark:text-gray-200 uppercase tracking-wider">WA</th>
                            <th class="py-3 px-4 text-left text-sm font-medium text-gray-700 dark:text-gray-200 uppercase tracking-wider text-center">Foto</th>
                            <th class="py-3 px-4 text-left text-sm font-medium text-gray-700 dark:text-gray-200 uppercase tracking-wider text-center">Tgl. Pembuatan</th>
                            <th class="py-3 px-4 text-left text-sm font-medium text-gray-700 dark:text-gray-200 uppercase tracking-wider text-center">Aksi</th>
                        </tr>
                    </thead>
                    <tbody id="supervisorTableBody" class="divide-y divide-gray-200 dark:divide-gray-700">
                        <?php echo renderSupervisorsTableRows($supervisors, $profilePicsDir); ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Powered by: digipas.id dengan logo dan tahun -->
        <div class="text-center mt-10">
            <img src="<?php echo $baseUploadDir; ?>digipas.png" alt="Logo Digipas.id" class="mx-auto mb-2 w-16 h-auto" onerror="this.onerror=null;this.src='https://placehold.co/64x16/AAAAAA/FFFFFF?text=Logo+Error';">
            <p class="text-sm text-gray-600 dark:text-gray-400">
                Powered by: digipas.id &copy; <span id="currentYear"></span>
            </p>
        </div>
    </main>

    <!-- Modal Tambah/Edit Pengawas -->
    <div id="supervisorModal" class="modal flex">
        <div class="modal-content glass-effect">
            <span class="close-button text-day-mode-black dark:text-white" onclick="closeModal()">&times;</span>
            <h2 id="modalTitle" class="text-2xl font-bold mb-6 text-day-mode-black dark:text-white">Tambah Pengawas</h2>
            <form id="supervisorForm" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post" enctype="multipart/form-data">
                <input type="hidden" name="action" id="formAction" value="add">
                <input type="hidden" name="id" id="supervisorId">
                <input type="hidden" name="current_foto" id="currentFoto">

                <div class="mb-4">
                    <label for="username" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Username</label>
                    <input type="text" id="username" name="username" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white">
                </div>
                <div class="mb-4">
                    <label for="email" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Email</label>
                    <input type="email" id="email" name="email" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white">
                </div>
                <div class="mb-4">
                    <label for="kata_sandi" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Kata Sandi <span id="passwordHint" class="text-xs text-gray-500 dark:text-gray-400">(Biarkan kosong jika tidak ingin mengubah)</span></label>
                    <input type="password" id="kata_sandi" name="kata_sandi"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white">
                </div>
                <div class="mb-4">
                    <label for="nama" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Nama Lengkap</label>
                    <input type="text" id="nama" name="nama" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white">
                </div>
                <div class="mb-4">
                    <label for="wa" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Nomor WhatsApp</label>
                    <input type="text" id="wa" name="wa" placeholder="Format: 628xxxxxxxxx" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white">
                </div>
                <div class="mb-4">
                    <label for="foto" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Foto Profil</label>
                    <input type="file" id="foto" name="foto" accept="image/*"
                           class="w-full text-gray-700 dark:text-gray-300 file:mr-4 file:py-2 file:px-4 file:rounded-md file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100">
                    <div id="fotoPreviewContainer" class="mt-2 hidden items-center space-x-3">
                        <span class="text-sm text-gray-600 dark:text-gray-400">Foto Saat Ini:</span>
                        <img id="currentFotoPreview" src="" alt="Current Photo" class="w-16 h-16 object-cover rounded-full border border-gray-300 dark:border-gray-600">
                        <label class="flex items-center text-sm text-red-600 dark:text-red-400 cursor-pointer">
                            <input type="checkbox" name="remove_foto" id="removeFotoCheckbox" value="1" class="mr-1"> Hapus Foto
                        </label>
                    </div>
                </div>

                <div class="flex justify-end mt-6">
                    <button type="button" onclick="closeModal()" class="px-4 py-2 bg-gray-300 text-gray-800 rounded-md shadow-sm hover:bg-gray-400 mr-2">Batal</button>
                    <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md shadow-sm hover:bg-blue-700">Simpan</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        const themeToggle = document.getElementById('themeToggle');
        const htmlElement = document.documentElement;
        const sunIcon = document.querySelector('.sun-icon');
        const moonIcon = document.querySelector('.moon-icon');
        
        // Fungsi untuk menerapkan tema (gelap atau terang)
        function applyTheme(theme) {
            if (theme === 'dark') {
                htmlElement.classList.add('dark');
                sunIcon.classList.add('hidden');
                moonIcon.classList.remove('hidden');
            } else {
                htmlElement.classList.remove('dark');
                sunIcon.classList.remove('hidden');
                moonIcon.classList.add('hidden');
            }
        }

        // Periksa preferensi tema yang tersimpan di localStorage saat halaman dimuat
        const savedTheme = localStorage.getItem('theme');
        if (savedTheme) {
            applyTheme(savedTheme);
        } else if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            applyTheme('dark');
        } else {
            applyTheme('light');
        }

        // Tambahkan event listener untuk tombol pengalih tema
        themeToggle.addEventListener('click', () => {
            if (htmlElement.classList.contains('dark')) {
                applyTheme('light');
                localStorage.setItem('theme', 'light');
            } else {
                localStorage.setItem('theme', 'dark');
                localStorage.setItem('theme', 'dark');
            }
        });

        // Set tahun saat ini
        document.getElementById('currentYear').textContent = new Date().getFullYear();

        // JavaScript untuk Dropdown Sidebar - Manajemen Pengguna
        const manajemenPenggunaDropdownToggle = document.getElementById('manajemenPenggunaDropdownToggle');
        const manajemenPenggunaSubmenu = document.getElementById('manajemenPenggunaSubmenu');
        const manajemenPenggunaDropdownArrow = document.getElementById('manajemenPenggunaDropdownArrow');

        if (manajemenPenggunaDropdownToggle && manajemenPenggunaSubmenu && manajemenPenggunaDropdownArrow) {
            manajemenPenggunaDropdownToggle.addEventListener('click', (e) => {
                e.preventDefault();
                manajemenPenggunaSubmenu.classList.toggle('hidden');
                manajemenPenggunaDropdownArrow.classList.toggle('rotate-180');
            });
            // Pastikan sub-menu manajemen pengguna terbuka jika sedang di halaman pengawas.php
            const currentPath = window.location.pathname;
            if (currentPath.includes('pengawas.php') || currentPath.includes('daftarsiswa.php')) {
                manajemenPenggunaSubmenu.classList.remove('hidden');
                manajemenPenggunaDropdownArrow.classList.add('rotate-180');
            }
        }


        // JavaScript untuk Dropdown Sidebar - Pengaturan
        const settingsDropdownToggle = document.getElementById('settingsDropdownToggle');
        const settingsSubmenu = document.getElementById('settingsSubmenu');
        const settingsDropdownArrow = document.getElementById('settingsDropdownArrow');

        if (settingsDropdownToggle && settingsSubmenu && settingsDropdownArrow) {
            settingsDropdownToggle.addEventListener('click', (e) => {
                e.preventDefault();
                settingsSubmenu.classList.toggle('hidden');
                settingsDropdownArrow.classList.toggle('rotate-180');
            });
            // Pastikan sub-menu pengaturan terbuka jika sedang berada di halaman waapi.php atau token.php
            const currentPath = window.location.pathname;
            if (currentPath.includes('waapi.php') || currentPath.includes('setting.php') || currentPath.includes('token.php')) {
                settingsSubmenu.classList.remove('hidden');
                settingsDropdownArrow.classList.add('rotate-180');
            }
        }

        // JavaScript untuk Sidebar Minimization
        const sidebarToggle = document.getElementById('sidebarToggle');
        const sidebar = document.getElementById('sidebar');
        const sidebarToggleIcon = document.querySelector('.sidebar-toggle-icon');

        // Fungsi untuk menginisialisasi sidebar berdasarkan status tersimpan
        function initializeSidebar() {
            const isMinimized = localStorage.getItem('sidebarMinimized') === 'true';
            if (isMinimized) {
                sidebar.classList.add('minimized');
                sidebarToggleIcon.setAttribute('d', 'M9 5l7 7-7 7'); // Chevron kanan
            } else {
                sidebar.classList.remove('minimized');
                sidebarToggleIcon.setAttribute('d', 'M4 6h16M4 12h16M4 18h16'); // Hamburger
            }
            // Tambahkan event listener setelah inisialisasi awal
            sidebarToggle.addEventListener('click', toggleSidebar);
        }
        
        // Fungsi untuk mengganti status sidebar dan menyimpannya
        function toggleSidebar() {
            const isCurrentlyMinimized = sidebar.classList.contains('minimized');
            if (isCurrentlyMinimized) {
                sidebar.classList.remove('minimized');
                sidebarToggleIcon.setAttribute('d', 'M4 6h16M4 12h16M4 18h16'); // Hamburger
                localStorage.setItem('sidebarMinimized', 'false');
            } else {
                sidebar.classList.add('minimized');
                sidebarToggleIcon.setAttribute('d', 'M9 5l7 7-7 7'); // Chevron kanan
                localStorage.setItem('sidebarMinimized', 'true');
            }
        }
        
        // Panggil inisialisasi sidebar saat DOM siap
        document.addEventListener('DOMContentLoaded', initializeSidebar);

        // --- JavaScript untuk Modal dan CRUD Pengawas ---
        const supervisorModal = document.getElementById('supervisorModal');
        const modalTitle = document.getElementById('modalTitle');
        const supervisorForm = document.getElementById('supervisorForm');
        const formAction = document.getElementById('formAction');
        const supervisorId = document.getElementById('supervisorId');
        const usernameInput = document.getElementById('username');
        const emailInput = document.getElementById('email');
        const kataSandiInput = document.getElementById('kata_sandi');
        const passwordHint = document.getElementById('passwordHint');
        const namaInput = document.getElementById('nama');
        const waInput = document.getElementById('wa');
        const fotoInput = document.getElementById('foto');
        const currentFoto = document.getElementById('currentFoto');
        const fotoPreviewContainer = document.getElementById('fotoPreviewContainer');
        const currentFotoPreview = document.getElementById('currentFotoPreview');
        const removeFotoCheckbox = document.getElementById('removeFotoCheckbox');

        function openModal() {
            supervisorModal.style.display = 'flex';
            // Reset form dan pesan error
            supervisorForm.reset();
            fotoPreviewContainer.classList.add('hidden');
            removeFotoCheckbox.checked = false;
        }

        function closeModal() {
            supervisorModal.style.display = 'none';
        }

        function openAddModal() {
            modalTitle.textContent = 'Tambah Pengawas Baru';
            formAction.value = 'add';
            kataSandiInput.required = true; // Password wajib diisi saat tambah
            passwordHint.classList.add('hidden'); // Sembunyikan hint "biarkan kosong"
            openModal();
        }

        function openEditModal(supervisor) {
            modalTitle.textContent = 'Edit Pengawas';
            formAction.value = 'edit';
            kataSandiInput.required = false; // Password tidak wajib diisi saat edit
            passwordHint.classList.remove('hidden'); // Tampilkan hint "biarkan kosong"

            supervisorId.value = supervisor.id;
            usernameInput.value = supervisor.username;
            emailInput.value = supervisor.email;
            namaInput.value = supervisor.nama;
            waInput.value = supervisor.wa;
            currentFoto.value = supervisor.foto; // Simpan nama foto saat ini

            if (supervisor.foto) {
                currentFotoPreview.src = `uploads/profile_pics/${supervisor.foto}`; // Path relatif ke foto
                fotoPreviewContainer.classList.remove('hidden');
            } else {
                fotoPreviewContainer.classList.add('hidden');
            }
            removeFotoCheckbox.checked = false; // Pastikan tidak tercentang saat membuka edit

            openModal();
        }

        function confirmDelete(id) {
            // Menggunakan modal kustom sebagai pengganti alert/confirm
            const confirmation = confirm("Anda yakin ingin menghapus pengawas ini?");
            if (confirmation) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = '<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>';
                
                const actionInput = document.createElement('input');
                actionInput.type = 'hidden';
                actionInput.name = 'action';
                actionInput.value = 'delete';
                form.appendChild(actionInput);

                const idInput = document.createElement('input');
                idInput.type = 'hidden';
                idInput.name = 'id_to_delete';
                idInput.value = id;
                form.appendChild(idInput);

                document.body.appendChild(form);
                form.submit();
            }
        }

        // Pencarian Real-time menggunakan AJAX
        let searchTimeout;
        function searchSupervisors() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(() => {
                const searchTerm = document.getElementById('supervisorSearch').value;
                const xhr = new XMLHttpRequest();
                xhr.open('GET', `<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>?search=${encodeURIComponent(searchTerm)}`);
                xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest'); // Tandai sebagai AJAX request
                xhr.onload = function() {
                    if (xhr.status === 200) {
                        document.getElementById('supervisorTableBody').innerHTML = xhr.responseText;
                    } else {
                        console.error('Error fetching supervisors:', xhr.statusText);
                    }
                };
                xhr.send();
            }, 300); // Debounce 300ms
        }

        // Event listener untuk menutup modal jika klik di luar konten modal
        window.onclick = function(event) {
            if (event.target == supervisorModal) {
                closeModal();
            }
        }
    </script>
</body>
</html>
