<?php
// Mulai sesi PHP. Ini penting untuk mengakses variabel sesi.
session_start();

// Sertakan file konfigurasi database
require_once '../config.php'; // Pastikan path ini benar

// Inisialisasi variabel pesan
$message = '';
$message_type = ''; // 'success' atau 'error'

// Periksa apakah pengguna sudah login
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== TRUE) {
    // Jika belum login, arahkan kembali ke halaman login
    header("location: index.php"); // Arahkan ke file login.php Anda
    exit;
}

// Periksa peran pengguna. Hanya admin yang bisa mengakses halaman ini.
if ($_SESSION['role'] !== 'admin') {
    // Jika bukan admin, arahkan ke halaman lain atau tampilkan pesan akses ditolak
    header("location: index.php"); // Contoh: arahkan ke halaman selamat datang umum
    exit;
}

// Data pengguna yang akan ditampilkan di halaman admin
$nama_pengguna = htmlspecialchars($_SESSION['nama']);
$username_pengguna = htmlspecialchars($_SESSION['username']);
$role_pengguna = htmlspecialchars($_SESSION['role']);

// Direktori untuk logo sekolah
$uploadDir = 'uploads/';

/**
 * Fungsi untuk mendapatkan pengaturan sekolah saat ini dari database.
 * @param mysqli $conn Objek koneksi database.
 * @return array Pengaturan sekolah (nama, alamat, logo) atau nilai default.
 */
function getSchoolSettings($conn) {
    $sql = "SELECT nama_sekolah, alamat_sekolah, logo_sekolah FROM pengaturan_sekolah WHERE id = 1 LIMIT 1";
    $result = $conn->query($sql);
    if ($result && $result->num_rows > 0) {
        return $result->fetch_assoc();
    }
    // Nilai default jika tidak ada pengaturan ditemukan
    return ['nama_sekolah' => 'Nama Sekolah Default', 'alamat_sekolah' => '', 'logo_sekolah' => ''];
}

/**
 * Fungsi untuk mendapatkan pengaturan WA API dari database.
 * @param mysqli $conn Objek koneksi database.
 * @return array Pengaturan WA API (token, secret_key, api_link) atau nilai default.
 */
function getWaApiSettings($conn) {
    $sql = "SELECT token, secret_key, api_link FROM wa_api_settings WHERE id = 1 LIMIT 1";
    $result = $conn->query($sql);
    // Setel nilai default untuk API link, akan ditimpa jika ada di database
    // Catatan: api_link tidak lagi digunakan di sendWhatsAppMessage, tetapi tetap diambil untuk konsistensi jika ada bagian lain yang membutuhkannya.
    $settings = ['token' => '', 'secret_key' => '', 'api_link' => 'https://jogja.wablas.com'];

    if ($result && $result->num_rows > 0) {
        $fetchedSettings = $result->fetch_assoc();
        // Timpa nilai default dengan nilai yang diambil dari database jika ada dan tidak kosong
        $settings['token'] = !empty($fetchedSettings['token']) ? $fetchedSettings['token'] : $settings['token'];
        $settings['secret_key'] = !empty($fetchedSettings['secret_key']) ? $fetchedSettings['secret_key'] : $settings['secret_key'];
        $settings['api_link'] = !empty($fetchedSettings['api_link']) ? $fetchedSettings['api_link'] : $settings['api_link'];
    }
    return $settings;
}

/**
 * Fungsi untuk mendapatkan pengaturan ujian saat ini dari database.
 * @param mysqli $conn Objek koneksi database.
 * @return array Pengaturan ujian (exam_token, last_generated_at, auto_generate_interval, interval_unit) atau nilai default.
 */
function getExamSettings($conn) {
    $sql = "SELECT exam_token, last_generated_at, auto_generate_interval, interval_unit FROM exam_settings WHERE id = 1 LIMIT 1";
    $result = $conn->query($sql);
    if ($result && $result->num_rows > 0) {
        $settings = $result->fetch_assoc();
        // Provide defaults if values are NULL or not set (e.g., for existing rows without these columns)
        $settings['exam_token'] = $settings['exam_token'] ?? 'Belum Diatur';
        $settings['last_generated_at'] = $settings['last_generated_at'] ?? null;
        $settings['auto_generate_interval'] = $settings['auto_generate_interval'] ?? 1; // Default 1 minute
        $settings['interval_unit'] = $settings['interval_unit'] ?? 'minutes'; // Default 'minutes'
        return $settings;
    }
    return [
        'exam_token' => 'Belum Diatur',
        'last_generated_at' => null,
        'auto_generate_interval' => 1, // Default 1 minute
        'interval_unit' => 'minutes' // Default 'minutes'
    ];
}

/**
 * Fungsi untuk menyimpan atau memperbarui pengaturan ujian di database.
 * @param mysqli $conn Objek koneksi database.
 * @param string $newToken Token ujian baru.
 * @param string|null $lastGeneratedAt Timestamp kapan token terakhir kali dibuat (YYYY-MM-DD HH:MM:SS).
 * @param int $autoGenerateInterval Interval generasi otomatis.
 * @param string $intervalUnit Unit interval (minutes/seconds).
 * @return bool|string True jika berhasil, string error jika gagal.
 */
function updateExamSettings($conn, $newToken, $lastGeneratedAt, $autoGenerateInterval, $intervalUnit) {
    $newToken = $conn->real_escape_string($newToken);
    $lastGeneratedAtStr = $lastGeneratedAt ? "'" . $conn->real_escape_string($lastGeneratedAt) . "'" : 'NULL';
    $autoGenerateInterval = (int)$autoGenerateInterval;
    $intervalUnit = $conn->real_escape_string($intervalUnit); // Sanitize new parameter

    $checkSql = "SELECT COUNT(*) as count FROM exam_settings WHERE id = 1";
    $checkResult = $conn->query($checkSql);
    $row = $checkResult->fetch_assoc();
    $count = $row['count'];

    if ($count > 0) {
        $sql = "UPDATE exam_settings SET exam_token = '$newToken', last_generated_at = $lastGeneratedAtStr, auto_generate_interval = $autoGenerateInterval, interval_unit = '$intervalUnit' WHERE id = 1";
    } else {
        $sql = "INSERT INTO exam_settings (id, exam_token, last_generated_at, auto_generate_interval, interval_unit) VALUES (1, '$newToken', $lastGeneratedAtStr, $autoGenerateInterval, '$intervalUnit')";
    }

    if ($conn->query($sql) === TRUE) {
        return true;
    } else {
        $error_message = $conn->error;
        error_log("Error updating exam settings: " . $error_message);
        return $error_message; // Return the specific database error
    }
}

/**
 * Fungsi untuk mendapatkan daftar pengawas beserta nomor telepon dari database.
 * @param mysqli $conn Objek koneksi database.
 * @return array Daftar pengawas, masing-masing dengan 'nama' dan 'phone_number'.
*/
function getSupervisors($conn) {
    // Mengambil nama dan nomor WhatsApp dari tabel 'pengguna' untuk role 'pengawas'
    $sql = "SELECT nama, wa FROM pengguna WHERE role = 'pengawas' AND wa IS NOT NULL AND wa != ''";
    $result = $conn->query($sql);
    $supervisors = [];
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $supervisors[] = ['nama' => $row['nama'], 'phone_number' => $row['wa']];
        }
    }
    return $supervisors;
}

/**
 * Fungsi untuk mengirim pesan WhatsApp menggunakan API eksternal dengan link API tetap.
 * @param string $token Token API.
 * @param string $secretKey Secret Key API.
 * @param string $phoneNumber Nomor telepon penerima.
 * @param string $messageContent Isi pesan yang akan dikirim.
 * @return array Hasil pengiriman (status: 'success'/'error', message: pesan detail).
 */
function sendWhatsAppMessage($token, $secretKey, $phoneNumber, $messageContent) {
    // Pastikan nomor telepon dimulai dengan kode negara (misal: 62 untuk Indonesia)
    if (!preg_match('/^62/', $phoneNumber)) {
        $phoneNumber = '62' . ltrim($phoneNumber, '0');
    }

    // URL API Wablas yang telah ditentukan
    $apiLink = "https://jogja.wablas.com";
    $apiLink = rtrim($apiLink, '/'); // Pastikan apiLink tidak diakhiri dengan '/' untuk menghindari double slash

    // Bangun URL sesuai format yang diberikan pengguna secara langsung pada curl_init
    $fullApiUrl = $apiLink . "/api/send-message?token=" . urlencode($token . '.' . $secretKey) . "&phone=" . urlencode($phoneNumber) . "&message=" . urlencode($messageContent);

    $ch = curl_init($fullApiUrl); // Inisialisasi cURL dengan URL lengkap
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); // Mengembalikan respons sebagai string
    curl_setopt($ch, CURLOPT_TIMEOUT, 10); // Tambahkan timeout 10 detik

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); // Ambil kode status HTTP
    $error = curl_error($ch); // Ambil error cURL jika ada
    curl_close($ch);

    if ($error) {
        error_log("WhatsApp API Error (cURL): " . $error);
        return ['status' => 'error', 'message' => 'Gagal mengirim pesan WhatsApp: ' . $error];
    } elseif ($httpCode !== 200) {
        // Log error HTTP dan respon mentah
        error_log("WhatsApp API HTTP Error: " . $httpCode . " (URL: " . $fullApiUrl . ") Response: " . $response);
        $decodedResponse = json_decode($response, true);
        $apiErrorMessage = isset($decodedResponse['message']) ? $decodedResponse['message'] : 'Tidak ada respon spesifik dari API.';
        return ['status' => 'error', 'message' => 'Gagal mengirim pesan WhatsApp. Status HTTP: ' . $httpCode . '. Respon API: ' . $apiErrorMessage];
    } else {
        $decodedResponse = json_decode($response, true);
        // Periksa struktur respons API Wablas yang sebenarnya.
        // Asumsi Wablas mengembalikan status: true (boolean) atau status: "success" (string)
        if (isset($decodedResponse['status']) && ($decodedResponse['status'] === true || $decodedResponse['status'] === 'success')) {
            return ['status' => 'success', 'message' => 'Pesan WhatsApp berhasil dikirim.'];
        } else {
            // Log respons API yang tidak terduga
            error_log("WhatsApp API Unexpected Response: " . $response);
            $apiErrorMessage = isset($decodedResponse['message']) ? $decodedResponse['message'] : 'Respon API tidak terduga atau tidak ada pesan spesifik.';
            return ['status' => 'error', 'message' => 'Gagal mengirim pesan WhatsApp. Respon API: ' . $apiErrorMessage];
        }
    }
}

/**
 * Fungsi untuk menghasilkan token acak.
 * @param int $length Panjang token yang diinginkan.
 * @return string Token acak.
 */
function generateRandomToken($length = 5) { // Panjang token diubah menjadi 5 karakter
    $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
    $charactersLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, $charactersLength - 1)];
    }
    return $randomString;
}

// Ambil pengaturan sekolah
$settings = getSchoolSettings($conn);
$nama_sekolah_display = htmlspecialchars($settings['nama_sekolah']);
$nama_sekolah_raw = $settings['nama_sekolah']; // Dapatkan nama sekolah tanpa htmlspecialchars untuk pesan WA

// --- START: Handle AJAX request for automatic token generation ---
if (isset($_GET['action']) && $_GET['action'] == 'generate_new_token') {
    header('Content-Type: application/json');

    // Security check: ensure admin is logged in
    if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== TRUE || $_SESSION['role'] !== 'admin') {
        echo json_encode(['status' => 'error', 'message' => 'Unauthorized access.']);
        exit;
    }

    $currentSettings = getExamSettings($conn);
    $autoGenerateInterval = (int)$currentSettings['auto_generate_interval'];
    $intervalUnit = $currentSettings['interval_unit']; // Get the unit from DB

    $newExamToken = generateRandomToken(5); // Pastikan panjang token 5 karakter
    $lastGeneratedAt = date('Y-m-d H:i:s');

    $updateResult = updateExamSettings($conn, $newExamToken, $lastGeneratedAt, $autoGenerateInterval, $intervalUnit);

    $wa_send_summary = ['sent' => 0, 'failed' => 0, 'details' => []];

    if ($updateResult === true) { // Check for boolean true
        // --- START: WA Sending Logic for AJAX auto-generation ---
        $schoolSettingsForWA = getSchoolSettings($conn); // Re-fetch school settings for WA message context
        $nama_sekolah_for_wa = $schoolSettingsForWA['nama_sekolah'];

        $waApiSettings = getWaApiSettings($conn);
        $apiToken = $waApiSettings['token'];
        $apiSecretKey = $waApiSettings['secret_key'];

        if (empty($apiToken) || empty($apiSecretKey)) {
            $wa_send_summary['details'][] = "Peringatan: Pengaturan WA API (Token atau Secret Key) belum lengkap. Pesan ke pengawas tidak dapat dikirim.";
        } else {
            $supervisors = getSupervisors($conn);

            if (empty($supervisors)) {
                $wa_send_summary['details'][] = "Peringatan: Tidak ada pengawas dengan nomor telepon yang terdaftar untuk dikirimi pesan WhatsApp.";
            } else {
                foreach ($supervisors as $supervisor) {
                    $supervisorName = htmlspecialchars($supervisor['nama']);
                    $supervisorPhone = htmlspecialchars($supervisor['phone_number']);

                    // Construct the new WhatsApp message content based on the user's requested format
                    $whatsappMessageContent = "📢 ADMIN *".$nama_sekolah_for_wa. "*\n\n";
                    $whatsappMessageContent .= "👋 HALO *" . $supervisorName . "*, TOKEN TELAH BERUBAH DENGAN TOKEN TERBARU.\n\n";
                    $whatsappMessageContent .= "🔑 Token Baru:\n";
                    $whatsappMessageContent .= "*" . $newExamToken . "*\n\n";
                    $whatsappMessageContent .= "📌 Catatan Penting:\n";
                    $whatsappMessageContent .= "1. Jangan ditulis di papan tulis.\n";
                    $whatsappMessageContent .= "2. Token hanya diberikan kepada siswa yang benar-benar meminta.\n";
                    $whatsappMessageContent .= "3. Jangan lupa catat di berita acara.\n\n";
                    $whatsappMessageContent .= "🙏 Terima kasih atas kerja keras Bapak/Ibu pengawas di *" . $nama_sekolah_for_wa . "*.\n\n";
                    $whatsappMessageContent .= "Salam hormat,\n";
                    $whatsappMessageContent .= "Tim IT *" . $nama_sekolah_for_wa . "*";

                    $sendResult = sendWhatsAppMessage($apiToken, $apiSecretKey, $supervisorPhone, $whatsappMessageContent);

                    if ($sendResult['status'] === 'success') {
                        $wa_send_summary['sent']++;
                    } else {
                        $wa_send_summary['failed']++;
                        $wa_send_summary['details'][] = "Gagal mengirim WA ke $supervisorName ($supervisorPhone): " . $sendResult['message'];
                    }
                }
            }
        }
        // --- END: WA Sending Logic for AJAX auto-generation ---

        echo json_encode([
            'status' => 'success',
            'token' => $newExamToken,
            'last_generated_at_timestamp' => strtotime($lastGeneratedAt),
            'interval' => $autoGenerateInterval,
            'interval_unit' => $intervalUnit,
            'wa_send_summary' => $wa_send_summary // Include WA send results in AJAX response
        ]);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Failed to generate and save new token: ' . $updateResult]);
    }
    exit;
}
// --- END: Handle AJAX request for automatic token generation ---


// Tangani pengiriman formulir (non-AJAX)
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $isAutoGenerate = isset($_POST['generate_automatically']) && $_POST['generate_automatically'] === 'on';
    $newExamToken = $_POST['exam_token'] ?? '';
    $autoGenerateInterval = (int)($_POST['auto_generate_interval'] ?? 1); // Dapatkan interval dari form
    $intervalUnit = $_POST['interval_unit'] ?? 'minutes'; // New: Get unit from form

    $lastGeneratedAt = null; // Defaultkan ke null

    // If auto-generate is active, a new token is generated only on form submission.
    // Continuous generation is handled by AJAX. This block primarily updates settings.
    if ($isAutoGenerate) {
        // If the form is submitted with auto-generate ON, we should generate a new token
        // and update its timestamp to reflect the latest manual action/update of settings.
        $newExamToken = generateRandomToken(5); // Pastikan panjang token 5 karakter
        $lastGeneratedAt = date('Y-m-d H:i:s');
    }

    $sendWaMessages = isset($_POST['send_wa_messages']) && $_POST['send_wa_messages'] === 'on'; // Periksa status toggle
    $wa_send_results = []; // Inisialisasi array untuk menyimpan hasil pengiriman WA ke setiap pengawas

    // Menggunakan fungsi updateExamSettings yang baru
    $updateResult = updateExamSettings($conn, $newExamToken, $lastGeneratedAt, $autoGenerateInterval, $intervalUnit);
    if ($updateResult === true) { // Check for boolean true
        $message = "Token ujian berhasil diperbarui!";
        $message_type = "success"; // Status awal untuk pembaruan token

        if ($sendWaMessages) { // Hanya coba kirim WA jika toggle aktif
            // Ambil pengaturan WA API (hanya untuk token dan secret key)
            $waApiSettings = getWaApiSettings($conn);
            $apiToken = $waApiSettings['token'];
            $apiSecretKey = $waApiSettings['secret_key'];

            if (empty($apiToken) || empty($apiSecretKey)) { // Cek hanya token dan secret key
                $message .= "<br>Peringatan: Pengaturan WA API (Token atau Secret Key) belum lengkap. Pesan ke pengawas tidak dapat dikirim.";
                // Jika ada peringatan penting, ubah jenis pesan menjadi error/warning
                $message_type = "error";
            } else {
                // Ambil daftar pengawas
                $supervisors = getSupervisors($conn);

                if (empty($supervisors)) {
                    $message .= "<br>Peringatan: Tidak ada pengawas dengan nomor telepon yang terdaftar untuk dikirimi pesan WhatsApp.";
                    $message_type = "error";
                } else {
                    $total_sent = 0;
                    $total_failed = 0;
                    $failed_details = [];

                    foreach ($supervisors as $supervisor) {
                        $supervisorName = htmlspecialchars($supervisor['nama']);
                        $supervisorPhone = htmlspecialchars($supervisor['phone_number']);
                        // Construct the new WhatsApp message content based on the user's requested format
                        $whatsappMessageContent = "📢 ADMIN *" . $nama_sekolah_raw . "*\n\n";
                        $whatsappMessageContent .= "👋 HALO *" . $supervisorName . "*, TOKEN TELAH BERUBAH DENGAN TOKEN TERBARU.\n\n";
                        $whatsappMessageContent .= "🔑 Token Baru:\n";
                        $whatsappMessageContent .= "*" . $newExamToken . "*\n\n";
                        $whatsappMessageContent .= "📌 Catatan Penting:\n";
                        $whatsappMessageContent .= "1. Jangan ditulis di papan tulis.\n";
                        $whatsappMessageContent .= "2. Token hanya diberikan kepada siswa yang benar-benar meminta.\n";
                        $whatsappMessageContent .= "3. Jangan lupa catat di berita acara.\n\n";
                        $whatsappMessageContent .= "🙏 Terima kasih atas kerja keras Bapak/Ibu pengawas di *" . $nama_sekolah_raw . "*.\n\n";
                        $whatsappMessageContent .= "Salam hormat,\n";
                        $whatsappMessageContent .= "Tim IT *" . $nama_sekolah_raw . "*";


                        // Panggil fungsi sendWhatsAppMessage tanpa parameter $apiLink
                        $sendResult = sendWhatsAppMessage($apiToken, $apiSecretKey, $supervisorPhone, $whatsappMessageContent);

                        // Simpan hasil untuk setiap pengiriman
                        $wa_send_results[] = [
                            'name' => $supervisorName,
                            'phone' => $supervisorPhone,
                            'status' => $sendResult['status'],
                            'message' => $sendResult['message']
                        ];

                        if ($sendResult['status'] === 'success') {
                            $total_sent++;
                        } else {
                            $total_failed++;
                            $failed_details[] = "Gagal mengirim WA ke $supervisorName ($supervisorPhone): " . $sendResult['message'];
                            // Jika ada satu saja yang gagal, tingkatkan message_type menjadi error
                            if ($message_type === "success") { // Hanya ubah jika belum error
                                $message_type = "error";
                            }
                        }
                    }

                    // Tambahkan ringkasan pengiriman WA ke pesan utama
                    if ($total_sent > 0 && $total_failed === 0) {
                        $message .= "<br>Pesan WhatsApp berhasil dikirim ke *$total_sent* pengawas.";
                    } elseif ($total_sent > 0 && $total_failed > 0) {
                        $message .= "<br>Pesan WhatsApp berhasil dikirim ke *$total_sent* pengawas, namun *$total_failed* gagal dikirim.";
                        $message .= "<br>Detail Kegagalan:<br>" . implode("<br>", $failed_details);
                    } elseif ($total_sent === 0 && $total_failed > 0) {
                        $message .= "<br>Gagal mengirim pesan WhatsApp ke semua pengawas yang terdaftar.";
                        $message .= "<br>Detail Kegagalan:<br>" . implode("<br>", $failed_details);
                    }
                }
            }
        } else {
            $message .= "<br>Pengiriman pesan WhatsApp dinonaktifkan.";
        }
    } else {
        $message = "Gagal memperbarui token ujian. Silakan coba lagi. Detail: " . $updateResult; // Show the specific database error
        $message_type = "error";
    }
}

// Ambil pengaturan sekolah
$settings = getSchoolSettings($conn);
$nama_sekolah_display = htmlspecialchars($settings['nama_sekolah']);
// Menggunakan placeholder jika logo tidak ada atau gagal dimuat
$logo_sekolah_path = !empty($settings['logo_sekolah']) ? $uploadDir . htmlspecialchars($settings['logo_sekolah']) : 'https://placehold.co/60x60/B0E0E6/000000?text=LOGO';

// --- START: PHP untuk pengaturan ujian dan passing ke JS ---
// Pada pemuatan halaman awal atau setelah POST, selalu ambil pengaturan terbaru
$currentExamSettings = getExamSettings($conn); // Menggunakan fungsi baru

$currentExamToken = htmlspecialchars($currentExamSettings['exam_token']);
$lastGeneratedAtTimestamp = $currentExamSettings['last_generated_at'] ? strtotime($currentExamSettings['last_generated_at']) : null;
$currentAutoGenerateInterval = (int)$currentExamSettings['auto_generate_interval'];
$currentIntervalUnit = htmlspecialchars($currentExamSettings['interval_unit']); // New: Unit

// Tentukan status awal toggle berdasarkan POST sebelumnya atau default
$isWaSendEnabledByDefault = true;
if ($_SERVER["REQUEST_METHOD"] == "POST" && (!isset($_POST['send_wa_messages']) || $_POST['send_wa_messages'] !== 'on')) {
    $isWaSendEnabledByDefault = false;
}

$isAutoGenerateEnabledByDefault = false; // Defaultnya mati, user harus mengaktifkan
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['generate_automatically']) && $_POST['generate_automatically'] === 'on') {
    $isAutoGenerateEnabledByDefault = true;
}
// --- END: PHP untuk pengaturan ujian dan passing ke JS ---
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pengaturan Token Ujian - <?php echo $nama_pengguna; ?></title>
    <!-- Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Font Inter dari Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            transition: background-color 0.5s ease-in-out, color 0.5s ease-in-out;
            overflow-x: hidden; /* Mencegah scroll horizontal */
        }
        /* Mode gelap: Latar belakang yang lebih dalam */
        html.dark body {
            background-color: #0b0c10;
            color: #e2e8f0;
        }
        /* Mode terang: Latar belakang yang lebih hidup */
        html:not(.dark) body {
            background-color: #e0e7ff;
            color: #1a202c;
        }
        /* Transisi untuk elemen di dalam kartu */
        .glass-effect {
            transition: background-color 0.5s ease-in-out, border-color 0.5s ease-in-out, box-shadow 0.5s ease-in-out;
            background-color: rgba(255, 255, 255, 0.8); /* Latar belakang terang */
            backdrop-filter: blur(10px); /* Efek blur */
            border: 1px solid rgba(255, 255, 255, 0.3); /* Border samar */
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        html.dark .glass-effect {
            background-color: rgba(30, 41, 59, 0.8); /* Latar belakang gelap */
            border: 1px solid rgba(45, 55, 72, 0.5);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.2);
        }
        /* Mode gelap untuk teks */
        html.dark .text-day-mode-black {
            color: #e2e8f0;
        }
        /* Mode terang untuk teks: tetap hitam */
        html:not(.dark) .text-day-mode-black {
            color: #000000;
        }
        html.dark .sidebar {
            background-color: #1a202c;
            border-right-color: #2d3748;
        }
        html:not(.dark) .sidebar {
            background-color: #ffffff;
            border-right-color: #e2e8f0;
        }
        html.dark .sidebar a {
            color: #a0aec0;
        }
        html.dark .sidebar a:hover {
            background-color: #2d3748;
            color: #ffffff;
        }
        html:not(.dark) .sidebar a {
            color: #4a5568;
        }
        html:not(.dark) .sidebar a:hover {
            background-color: #edf2f7;
            color: #2d3748;
        }
        /* Gaya untuk tombol pengalih tema */
        #themeToggle {
            transition: background-color 0.5s ease-in-out, color 0.5s ease-in-out, box-shadow 0.5s ease-in-out;
        }

        /* Gaya untuk sidebar minimisasi */
        .sidebar {
            width: 256px; /* Lebar default (w-64) */
            transition: width 0.3s ease-in-out; /* Transisi untuk perubahan lebar */
            flex-shrink: 0; /* Mencegah sidebar menyusut saat konten utama lebar */
        }
        .sidebar.minimized {
            width: 64px; /* Lebar saat diminimalisir (w-16) */
        }
        .sidebar.minimized .sidebar-text {
            display: none; /* Sembunyikan teks menu */
        }
        .sidebar.minimized .sidebar-icon {
            margin-right: 0 !important; /* Hapus margin dari ikon */
            /* Mengatur ukuran ikon agar terlihat jelas saat diminimalisir */
            width: 24px; /* Sedikit lebih besar dari w-5 (20px) */
            height: 24px; /* Sedikit lebih besar dari h-5 (20px) */
        }
        .sidebar.minimized .sidebar-menu-item {
            justify-content: center; /* Pusatkan ikon */
            padding-left: 0; /* Hapus padding kiri */
            padding-right: 0; /* Hapus padding kanan */
        }
        .sidebar.minimized .sidebar-dropdown-icon {
            display: none; /* Sembunyikan ikon panah dropdown saat diminimalisir */
        }
        .sidebar.minimized .sidebar-header {
            visibility: hidden; /* Sembunyikan teks header sidebar */
            white-space: nowrap; /* Pastikan teks tidak membungkus */
            overflow: hidden; /* Sembunyikan overflow */
            width: 0; /* Atur lebar menjadi nol */
            padding-left: 0;
            padding-right: 0;
        }
        .sidebar.minimized .sidebar-toggle-button {
            justify-content: center; /* Pusatkan tombol toggle */
        }
        .sidebar.minimized .school-logo {
            margin-right: 0 !important; /* Hapus margin kanan logo */
            width: 40px; /* Sesuaikan ukuran logo saat diminimalisir */
            height: 40px; /* Sesuaikan ukuran logo saat diminimalisir */
        }
        .sidebar.minimized .school-name {
            display: none; /* Sembunyikan nama sekolah saat diminimalisir (tetap) */
        }

        /* Tooltip untuk ikon saat sidebar diminimalisir */
        .sidebar.minimized [data-tooltip]:hover:after {
            content: attr(data-tooltip); /* Ambil teks dari atribut data-tooltip */
            position: absolute;
            left: 70px; /* Sesuaikan posisi tooltip */
            top: 50%;
            transform: translateY(-50%);
            background-color: #333;
            color: #fff;
            padding: 5px 10px;
            border-radius: 4px;
            white-space: nowrap;
            z-index: 999;
            opacity: 1;
            transition: opacity 0.2s ease;
        }
        .sidebar.minimized [data-tooltip]:after {
            opacity: 0; /* Sembunyikan secara default */
            pointer-events: none; /* Jangan menghalangi interaksi mouse */
        }

        /* Styles for the custom toggle switch */
        .toggle-switch {
            position: relative;
            display: inline-block;
            width: 60px;
            height: 34px;
        }

        .toggle-switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }

        .slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            -webkit-transition: .4s;
            transition: .4s;
            border-radius: 34px;
        }

        .slider:before {
            position: absolute;
            content: "";
            height: 26px;
            width: 26px;
            left: 4px;
            bottom: 4px;
            background-color: white;
            -webkit-transition: .4s;
            transition: .4s;
            border-radius: 50%;
        }

        input:checked + .slider {
            background-color: #2196F3;
        }

        html.dark input:checked + .slider {
            background-color: #60a5fa; /* Lighter blue for dark mode */
        }

        input:focus + .slider {
            box-shadow: 0 0 1px #2196F3;
        }

        input:checked + .slider:before {
            -webkit-transform: translateX(26px);
            -ms-transform: translateX(26px);
            transform: translateX(26px);
        }
    </style>
</head>
<body class="flex min-h-screen">

    <!-- Sidebar -->
    <aside id="sidebar" class="sidebar w-64 p-5 border-r shadow-lg flex flex-col justify-between transition-colors duration-500 relative">
        <div>
            <div class="flex items-center justify-between mb-8">
                <div class="flex items-center sidebar-header">
                    <!-- Logo Sekolah -->
                    <img src="<?php echo $logo_sekolah_path; ?>" alt="Logo Sekolah" class="school-logo w-12 h-12 object-contain mr-3 rounded-md" onerror="this.onerror=null;this.src='https://placehold.co/60x60/B0E0E6/000000?text=LOGO';">
                    <span class="text-xl font-bold text-blue-600 dark:text-blue-400 school-name"><?php echo $nama_sekolah_display; ?></span>
                </div>
                <button id="sidebarToggle" class="p-2 rounded-full hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-toggle-button">
                    <!-- Icon untuk toggle sidebar (hamburger atau chevron) -->
                    <svg class="w-6 h-6 sidebar-toggle-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                    </svg>
                </button>
            </div>
            <nav>
                <ul class="space-y-1">
                    <li>
                        <a href="admin.php" class="flex items-center py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Dashboard">
                            <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"></path></svg>
                            <span class="sidebar-text">Dashboard</span>
                        </a>
                    </li>
                    <!-- Dropdown Menu untuk Manajemen Pengguna -->
                    <li class="relative">
                        <a href="#" id="manajemenPenggunaDropdownToggle" class="flex items-center justify-between py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Manajemen Pengguna">
                            <div class="flex items-center">
                                <!-- Icon "users" -->
                                <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h-5v-9H4L12 3l8 8h-8v9z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 11c-2.209 0-4-1.791-4-4s1.791-4 4-4 4 1.791 4 4-1.791 4-4 4zM12 13a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path></svg>
                                <span class="sidebar-text">Manajemen Pengguna</span>
                            </div>
                            <svg id="manajemenPenggunaDropdownArrow" class="w-4 h-4 sidebar-dropdown-icon transform transition-transform duration-200" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </a>
                        <ul id="manajemenPenggunaSubmenu" class="pl-8 mt-1 space-y-1 hidden">
                            <li>
                                <a href="daftarsiswa.php" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Lihat Semua Pengguna">
                                    <span class="sidebar-text">Daftar Siswa</span>
                                </a>
                            </li>
                            <li>
                                <a href="pengawas.php" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Manajemen Pengawas">
                                    <span class="sidebar-text">Manajemen Pengawas</span>
                                </a>
                            </li>
                            <li>
                                <a href="#" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Tambah Pengguna Baru">
                                    <span class="sidebar-text">Tambah Pengguna</span>
                                </a>
                            </li>
                        </ul>
                    </li>
                    <li>
                        <a href="#" class="flex items-center py-2 px-3 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Laporan">
                            <!-- Icon "book" -->
                            <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.523 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5s3.332.477 4.5 1.253v13C19.832 18.523 18.246 18 16.5 18s-3.332.477-4.5 1.253"></path></svg>
                            <span class="sidebar-text">Laporan</span>
                        </a>
                    </li>
                    <!-- Dropdown Menu untuk Pengaturan -->
                    <li class="relative">
                        <a href="#" id="settingsDropdownToggle" class="flex items-center justify-between py-2 px-3 rounded-lg bg-gray-100 dark:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Pengaturan">
                            <div class="flex items-center">
                                <!-- Icon "settings" -->
                                <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.525.322 1.018.673 1.505 1.01z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path></svg>
                                <span class="sidebar-text">Pengaturan</span>
                            </div>
                            <svg id="settingsDropdownArrow" class="w-4 h-4 sidebar-dropdown-icon transform transition-transform duration-200" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </a>
                        <ul id="settingsSubmenu" class="pl-8 mt-1 space-y-1 hidden">
                            <li>
                                <a href="setting.php" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Pengaturan Umum">
                                    <span class="sidebar-text">Pengaturan Umum</span>
                                </a>
                            </li>
                            <li>
                                <a href="waapi.php" class="flex items-center py-1 px-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors duration-200 sidebar-menu-item" data-tooltip="Pengaturan WA API">
                                    <span class="sidebar-text">Pengaturan WA API</span>
                                </a>
                            </li>
                            <li>
                                <a href="token.php" class="flex items-center py-1 px-2 rounded-lg bg-gray-200 dark:bg-gray-800 transition-colors duration-200 sidebar-menu-item" data-tooltip="Pengaturan Token Ujian">
                                    <span class="sidebar-text">Token Ujian</span>
                                </a>
                            </li>
                        </ul>
                    </li>
                </ul>
            </nav>
        </div>
        <div class="mt-8">
            <a href="logout.php" class="flex items-center py-2 px-3 rounded-lg bg-red-500 hover:bg-red-600 text-white transition-colors duration-200 sidebar-menu-item" data-tooltip="Logout">
                <svg class="w-5 h-5 mr-3 sidebar-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path></svg>
                <span class="sidebar-text">Logout</span>
            </a>
        </div>
    </aside>

    <!-- Konten Utama -->
    <main id="mainContent" class="flex-1 p-10 transition-all duration-300">
        <!-- Tombol pengalih tema gelap/terang -->
        <div class="absolute top-4 right-4 z-10">
            <button id="themeToggle" class="p-2 rounded-full bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white shadow-md">
                <!-- Icon SVG untuk tema terang (matahari) -->
                <svg class="w-6 h-6 sun-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 3v1m0 16v1m9-9h1M3 12h1m15.325-7.757l-.707-.707M4.343 19.657l-.707-.707m12.728 0l-.707.707M6.343 6.343l-.707-.707"></path>
                </svg>
                <!-- Ikon bulan, tersembunyi secara default -->
                <svg class="w-6 h-6 moon-icon hidden" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9 9 0 008.354-5.646z"></path>
                </svg>
            </button>
        </div>

        <h1 class="text-4xl font-extrabold mb-8 text-day-mode-black dark:text-white">Pengaturan Token Ujian</h1>

        <?php if (!empty($message)): ?>
            <div class="p-4 mb-4 rounded-lg
                <?php echo $message_type === 'success' ? 'bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-100' : 'bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-100'; ?>
                shadow-md" role="alert">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div class="glass-effect p-8 rounded-lg shadow-xl mb-8">
            <h3 class="2xl font-semibold mb-6 text-day-mode-black dark:text-white">Kelola Token Ujian</h3>
            <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post">
                <div class="mb-5">
                    <label for="exam_token" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Token Ujian</label>
                    <input type="text" id="exam_token" name="exam_token" value="<?php echo $currentExamToken; ?>"
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white dark:focus:ring-blue-400 dark:focus:border-blue-400"
                            placeholder="Masukkan token ujian baru" required>
                </div>

                <div class="mb-5 flex items-center justify-between">
                    <label for="generate_automatically" class="text-sm font-medium text-gray-700 dark:text-gray-300">Buat Token Otomatis</label>
                    <label class="toggle-switch">
                        <input type="checkbox" id="generate_automatically" name="generate_automatically" value="on" <?php echo $isAutoGenerateEnabledByDefault ? 'checked' : ''; ?>>
                        <span class="slider"></span>
                    </label>
                </div>

                <div id="autoGenerateOptions" class="mb-5 <?php echo $isAutoGenerateEnabledByDefault ? '' : 'hidden'; ?>">
                    <div class="flex items-end space-x-3">
                        <div class="flex-1">
                            <label for="auto_generate_interval" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Interval Generasi Otomatis</label>
                            <input type="number" id="auto_generate_interval" name="auto_generate_interval" value="<?php echo $currentAutoGenerateInterval; ?>" min="1"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white dark:focus:ring-blue-400 dark:focus:border-blue-400"
                                   required>
                        </div>
                        <div>
                            <label for="interval_unit" class="sr-only">Unit Interval</label>
                            <select id="interval_unit" name="interval_unit"
                                    class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white dark:focus:ring-blue-400 dark:focus:border-blue-400">
                                <option value="minutes" <?php echo ($currentIntervalUnit === 'minutes') ? 'selected' : ''; ?>>Menit</option>
                                <option value="seconds" <?php echo ($currentIntervalUnit === 'seconds') ? 'selected' : ''; ?>>Detik</option>
                            </select>
                        </div>
                    </div>
                    <p id="countdownDisplay" class="text-sm text-gray-500 dark:text-gray-400 mt-2"></p>
                </div>

                <div class="mb-5 flex items-center justify-between">
                    <label for="send_wa_messages" class="text-sm font-medium text-gray-700 dark:text-gray-300">Kirim Pesan WhatsApp ke Pengawas</label>
                    <label class="toggle-switch">
                        <input type="checkbox" id="send_wa_messages" name="send_wa_messages" value="on" <?php echo $isWaSendEnabledByDefault ? 'checked' : ''; ?>>
                        <span class="slider"></span>
                    </label>
                </div>

                <div class="flex justify-end">
                    <button type="submit" class="px-6 py-3 bg-blue-600 text-white font-semibold rounded-lg shadow-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-75 transition-colors duration-200">
                        Perbarui & Kirim Token
                    </button>
                </div>
            </form>
        </div>

        <!-- Powered by: digipas.id dengan logo dan tahun -->
        <div class="text-center mt-10">
            <!-- Updated placeholder for digipas.png -->
            <img src="https://placehold.co/64x16/AAAAAA/FFFFFF?text=DIGIPAS.ID" alt="Logo Digipas.id" class="mx-auto mb-2 w-16 h-auto" onerror="this.onerror=null;this.src='https://placehold.co/64x16/AAAAAA/FFFFFF?text=DIGIPAS.ID';">
            <p class="text-sm text-gray-600 dark:text-gray-400">
                Powered by: digipas.id &copy; <span id="currentYear"></span>
            </p>
        </div>
    </main>

    <!-- Skrip JavaScript -->
    <script>
        const themeToggle = document.getElementById('themeToggle');
        const htmlElement = document.documentElement;
        const sunIcon = document.querySelector('.sun-icon');
        const moonIcon = document.querySelector('.moon-icon');

        // Fungsi untuk menerapkan tema (gelap atau terang)
        function applyTheme(theme) {
            if (theme === 'dark') {
                htmlElement.classList.add('dark');
                sunIcon.classList.add('hidden');
                moonIcon.classList.remove('hidden');
            } else {
                htmlElement.classList.remove('dark');
                sunIcon.classList.remove('hidden');
                moonIcon.classList.add('hidden');
            }
        }

        // Periksa preferensi tema yang tersimpan di localStorage saat halaman dimuat
        const savedTheme = localStorage.getItem('theme');
        if (savedTheme) {
            applyTheme(savedTheme);
        } else if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            applyTheme('dark');
        } else {
            applyTheme('light');
        }

        // Tambahkan event listener untuk tombol pengalih tema
        themeToggle.addEventListener('click', () => {
            if (htmlElement.classList.contains('dark')) {
                applyTheme('light');
                localStorage.setItem('theme', 'light');
            } else {
                localStorage.setItem('theme', 'dark');
                applyTheme('dark');
            }
        });

        // Set tahun saat ini
        document.getElementById('currentYear').textContent = new Date().getFullYear();

        // JavaScript untuk Dropdown Sidebar - Manajemen Pengguna
        const manajemenPenggunaDropdownToggle = document.getElementById('manajemenPenggunaDropdownToggle');
        const manajemenPenggunaSubmenu = document.getElementById('manajemenPenggunaSubmenu');
        const manajemenPenggunaDropdownArrow = document.getElementById('manajemenPenggunaDropdownArrow');

        if (manajemenPenggunaDropdownToggle && manajemenPenggunaSubmenu && manajemenPenggunaDropdownArrow) {
            manajemenPenggunaDropdownToggle.addEventListener('click', (e) => {
                e.preventDefault();
                manajemenPenggunaSubmenu.classList.toggle('hidden');
                manajemenPenggunaDropdownArrow.classList.toggle('rotate-180');
            });
             // Periksa jika salah satu sub-menu aktif agar dropdown terbuka
             const currentPath = window.location.pathname;
            if (currentPath.includes('daftarsiswa.php') || currentPath.includes('pengawas.php') || currentPath.includes('tambahpengguna.php')) {
                manajemenPenggunaSubmenu.classList.remove('hidden');
                manajemenPenggunaDropdownArrow.classList.add('rotate-180');
            }
        }

        // JavaScript untuk Dropdown Sidebar - Pengaturan
        const settingsDropdownToggle = document.getElementById('settingsDropdownToggle');
        const settingsSubmenu = document.getElementById('settingsSubmenu');
        const settingsDropdownArrow = document.getElementById('settingsDropdownArrow');

        if (settingsDropdownToggle && settingsSubmenu && settingsDropdownArrow) {
            settingsDropdownToggle.addEventListener('click', (e) => {
                e.preventDefault();
                settingsSubmenu.classList.toggle('hidden');
                settingsDropdownArrow.classList.toggle('rotate-180');
            });
            // Pastikan sub-menu pengaturan terbuka jika sedang berada di halaman waapi.php atau token.php
            const currentPath = window.location.pathname;
            if (currentPath.includes('waapi.php') || currentPath.includes('setting.php') || currentPath.includes('token.php')) {
                settingsSubmenu.classList.remove('hidden');
                settingsDropdownArrow.classList.add('rotate-180');
            }
        }

        // JavaScript untuk Sidebar Minimization
        const sidebarToggle = document.getElementById('sidebarToggle');
        const sidebar = document.getElementById('sidebar');
        const sidebarToggleIcon = document.querySelector('.sidebar-toggle-icon');

        // Fungsi untuk menginisialisasi sidebar berdasarkan status tersimpan
        function initializeSidebar() {
            const isMinimized = localStorage.getItem('sidebarMinimized') === 'true';
            if (isMinimized) {
                sidebar.classList.add('minimized');
                sidebarToggleIcon.setAttribute('d', 'M9 5l7 7-7 7'); // Chevron kanan
            } else {
                sidebar.classList.remove('minimized');
                sidebarToggleIcon.setAttribute('d', 'M4 6h16M4 12h16M4 18h16'); // Hamburger
            }
            // Tambahkan event listener setelah inisialisasi awal
            sidebarToggle.addEventListener('click', toggleSidebar);
        }

        // Fungsi untuk mengganti status sidebar dan menyimpannya
        function toggleSidebar() {
            const isCurrentlyMinimized = sidebar.classList.contains('minimized');
            if (isCurrentlyMinimized) {
                sidebar.classList.remove('minimized');
                sidebarToggleIcon.setAttribute('d', 'M4 6h16M4 12h16M4 18h16'); // Hamburger
                localStorage.setItem('sidebarMinimized', 'false');
            } else {
                sidebar.classList.add('minimized');
                sidebarToggleIcon.setAttribute('d', 'M9 5l7 7-7 7'); // Chevron kanan
                localStorage.setItem('sidebarMinimized', 'true');
            }
        }

        // Panggil inisialisasi sidebar saat DOM siap
        document.addEventListener('DOMContentLoaded', initializeSidebar);


        // --- START: JavaScript untuk mengelola input token dan toggle otomatis ---
        const examTokenInput = document.getElementById('exam_token');
        const generateAutomaticallyToggle = document.getElementById('generate_automatically');
        const autoGenerateOptionsDiv = document.getElementById('autoGenerateOptions');
        const autoGenerateIntervalInput = document.getElementById('auto_generate_interval');
        const intervalUnitSelect = document.getElementById('interval_unit'); // New: get unit select
        const countdownDisplay = document.getElementById('countdownDisplay');

        let countdownTimer; // Untuk menyimpan ID interval untuk dibersihkan

        // Fungsi format waktu untuk hitungan mundur
        function formatTime(seconds) {
            if (seconds < 0) return "Siap untuk generasi token baru!";
            const minutes = Math.floor(seconds / 60);
            const remainingSeconds = seconds % 60;
            return `Token baru dalam: ${minutes.toString().padStart(2, '0')}m ${remainingSeconds.toString().padStart(2, '0')}s`;
        }

        // Fungsi untuk memicu pembuatan token baru secara otomatis melalui AJAX
        async function triggerAutoGenerateToken() {
            countdownDisplay.textContent = 'Membuat token baru...';
            try {
                const response = await fetch('token.php?action=generate_new_token', {
                    method: 'GET',
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest' // Menandai sebagai AJAX request
                    }
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const data = await response.json();

                if (data.status === 'success') {
                    examTokenInput.value = data.token;
                    // Update initialLastGeneratedAt dengan timestamp baru dari server
                    initialLastGeneratedAt = data.last_generated_at_timestamp;
                    // Pastikan interval juga diperbarui jika ada perubahan dari sisi server
                    autoGenerateIntervalInput.value = data.interval;
                    intervalUnitSelect.value = data.interval_unit; // Update unit from server
                    startCountdown(); // Mulai hitungan mundur lagi dengan timestamp baru

                    // Display WhatsApp sending summary
                    let waSummaryMessage = "";
                    if (data.wa_send_summary) {
                        if (data.wa_send_summary.sent > 0 && data.wa_send_summary.failed === 0) {
                            waSummaryMessage = `Pesan WA berhasil dikirim ke ${data.wa_send_summary.sent} pengawas.`;
                        } else if (data.wa_send_summary.sent > 0 && data.wa_send_summary.failed > 0) {
                            waSummaryMessage = `Pesan WA berhasil dikirim ke ${data.wa_send_summary.sent} pengawas, namun ${data.wa_send_summary.failed} gagal. Detail: ${data.wa_send_summary.details.join(', ')}`;
                        } else if (data.wa_send_summary.failed > 0) {
                            waSummaryMessage = `Gagal mengirim pesan WA ke semua pengawas. Detail: ${data.wa_send_summary.details.join(', ')}`;
                        } else {
                             waSummaryMessage = `Peringatan: Tidak ada pengaturan WA API atau pengawas.`;
                        }
                    }
                    // This is a simple way to display; consider a more robust message box if needed
                    console.log("WhatsApp Send Summary:", waSummaryMessage);
                    // Optionally update a dedicated message div for WA status
                    // Example: document.getElementById('waStatusMessage').textContent = waSummaryMessage;

                } else {
                    countdownDisplay.textContent = `Gagal membuat token otomatis: ${data.message}`;
                    console.error('Error auto-generating token:', data.message);
                }
            } catch (error) {
                countdownDisplay.textContent = `Terjadi kesalahan jaringan: ${error.message}`;
                console.error('Fetch error:', error);
            }
        }

        // Fungsi untuk memulai hitungan mundur
        function startCountdown() {
            if (countdownTimer) {
                clearInterval(countdownTimer); // Hentikan timer sebelumnya jika ada
            }

            const intervalValue = parseInt(autoGenerateIntervalInput.value);
            const intervalUnit = intervalUnitSelect.value; // Get selected unit

            if (isNaN(intervalValue) || intervalValue <= 0) {
                countdownDisplay.textContent = "Interval tidak valid.";
                return;
            }

            let totalMilliseconds;
            if (intervalUnit === 'minutes') {
                totalMilliseconds = intervalValue * 60 * 1000;
            } else { // seconds
                totalMilliseconds = intervalValue * 1000;
            }

            // Konversi timestamp PHP (detik) ke milidetik untuk Date JavaScript
            const lastGenTime = initialLastGeneratedAt ? new Date(initialLastGeneratedAt * 1000) : null;
            const now = new Date();

            let targetTime;
            if (lastGenTime) {
                targetTime = new Date(lastGenTime.getTime() + totalMilliseconds);
                // Jika waktu target sudah lewat, hitung dari sekarang + interval
                if (targetTime < now) {
                    // Jika waktu terakhir generasi sudah lewat, hitung target berikutnya dari waktu sekarang
                    // Ini memastikan hitungan mundur dimulai dari interval penuh jika terlambat.
                    const elapsed = now.getTime() - lastGenTime.getTime();
                    const numIntervalsPassed = Math.floor(elapsed / totalMilliseconds);
                    targetTime = new Date(lastGenTime.getTime() + (numIntervalsPassed + 1) * totalMilliseconds);
                }
            } else {
                // Jika belum pernah ada generasi sebelumnya, hitung dari sekarang + interval
                targetTime = new Date(now.getTime() + totalMilliseconds);
            }

            // Fungsi untuk memperbarui tampilan hitungan mundur
            function updateCountdown() {
                const currentTime = new Date().getTime();
                let timeDifference = targetTime.getTime() - currentTime;
                let remainingSeconds = Math.max(0, Math.floor(timeDifference / 1000));

                countdownDisplay.textContent = formatTime(remainingSeconds);

                if (remainingSeconds <= 0) {
                    clearInterval(countdownTimer); // Hentikan timer
                    countdownDisplay.textContent = "Siap untuk generasi token baru!";
                    examTokenInput.value = 'Membuat token baru...'; // Tunjukkan status loading
                    triggerAutoGenerateToken(); // Panggil fungsi untuk membuat token baru
                }
            }

            updateCountdown(); // Panggil segera untuk menghindari penundaan awal
            countdownTimer = setInterval(updateCountdown, 1000); // Perbarui setiap detik
        }

        // Fungsi untuk mengelola status input token dan visibilitas opsi otomatis
        function toggleExamTokenInput() {
            if (generateAutomaticallyToggle.checked) {
                examTokenInput.disabled = true;
                examTokenInput.value = 'Akan dibuat otomatis...'; // Placeholder saat otomatis
                examTokenInput.setAttribute('placeholder', 'Akan dibuat otomatis...');

                autoGenerateOptionsDiv.classList.remove('hidden'); // Tampilkan opsi interval
                autoGenerateIntervalInput.disabled = false;
                intervalUnitSelect.disabled = false; // Enable the select
                startCountdown(); // Mulai hitungan mundur
            } else {
                examTokenInput.disabled = false;
                // Pulihkan nilai token sebenarnya jika sebelumnya diatur ke placeholder
                if (examTokenInput.value === 'Akan dibuat otomatis...' || examTokenInput.value === 'Siap untuk generasi token baru!' || examTokenInput.value === 'Membuat token baru...') {
                    examTokenInput.value = initialExamToken;
                }
                examTokenInput.setAttribute('placeholder', 'Masukkan token ujian baru');

                autoGenerateOptionsDiv.classList.add('hidden'); // Sembunyikan opsi interval
                autoGenerateIntervalInput.disabled = true;
                intervalUnitSelect.disabled = true; // Disable the select
                if (countdownTimer) {
                    clearInterval(countdownTimer); // Hentikan hitungan mundur
                }
                countdownDisplay.textContent = ''; // Hapus teks hitungan mundur
            }
        }

        // --- Variabel JavaScript dari PHP (Diposisikan di sini agar berada di dalam script tag) ---
        let initialExamToken = <?php echo json_encode($currentExamToken); ?>;
        let initialLastGeneratedAt = <?php echo json_encode($lastGeneratedAtTimestamp); ?>; // Unix timestamp atau null
        let initialAutoGenerateInterval = <?php echo json_encode($currentAutoGenerateInterval); ?>;
        let initialIntervalUnit = <?php echo json_encode($currentIntervalUnit); ?>; // New
        const initialIsAutoGenerateEnabled = <?php echo json_encode($isAutoGenerateEnabledByDefault); ?>;
        // --- END: Variabel JavaScript dari PHP ---

        // Atur status awal input saat DOM siap
        document.addEventListener('DOMContentLoaded', () => {
            // Pastikan nilai input interval mencerminkan nilai PHP
            autoGenerateIntervalInput.value = initialAutoGenerateInterval;
            intervalUnitSelect.value = initialIntervalUnit; // Set initial unit
            // Atur status toggle "Buat Token Otomatis" berdasarkan nilai PHP
            generateAutomaticallyToggle.checked = initialIsAutoGenerateEnabled;
            // Panggil fungsi toggle untuk menerapkan status awal ke input dan hitungan mundur
            toggleExamTokenInput();
        });

        // Tambahkan event listener untuk toggle "Buat Token Otomatis"
        generateAutomaticallyToggle.addEventListener('change', toggleExamTokenInput);
        // Tambahkan event listener untuk input interval agar hitungan mundur di-restart jika interval berubah
        autoGenerateIntervalInput.addEventListener('change', startCountdown);
        // New: Add event listener for interval unit change
        intervalUnitSelect.addEventListener('change', startCountdown);
        // --- END: JavaScript untuk mengelola input token dan toggle otomatis ---
    </script>
</body>
</html>
