<?php
session_start();

// 1. Konfigurasi & Keamanan
require_once '../config.php'; 

if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== TRUE || $_SESSION['role'] !== 'admin') {
    header("location: index.php");
    exit;
}

$nama_pengguna = htmlspecialchars($_SESSION['nama']);
$message = "";
$message_type = "";

// --- FUNGSI BANTUAN (Didefinisikan di atas agar bisa dipakai AJAX) ---
function getTimeStatus($last_activity) {
    if (empty($last_activity)) {
        return ['status' => 'offline', 'text' => 'Offline', 'time' => ''];
    }
    $time_diff = time() - strtotime($last_activity);
    if ($time_diff < 300) { // 5 menit
        return ['status' => 'online', 'text' => 'Online', 'time' => 'Baru saja'];
    } else {
        $minutes = floor($time_diff / 60);
        if ($minutes < 60) { $time_text = $minutes . " menit lalu"; } 
        elseif ($minutes < 1440) { $hours = floor($minutes / 60); $time_text = $hours . " jam lalu"; } 
        else { $days = floor($minutes / 1440); $time_text = $days . " hari lalu"; }
        return ['status' => 'offline', 'text' => 'Offline', 'time' => $time_text];
    }
}

// 2. LOGIKA ACTION (Import, Edit, Hapus, Reset) - Hanya dijalankan jika POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // ... (Logika Import Data CSV) ...
    if (isset($_POST['import_data'])) {
        $fileName = $_FILES['file_csv']['name'];
        $fileTmp = $_FILES['file_csv']['tmp_name'];
        $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

        if ($fileExt === 'csv') {
            $handle = fopen($fileTmp, "r");
            $row = 0; $success_count = 0;
            while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
                $row++; if ($row == 1) continue;
                $id_siswa = htmlspecialchars($data[0] ?? '');
                $qr_content = htmlspecialchars($data[1] ?? ''); 
                $nama = htmlspecialchars($data[2] ?? '');
                $kelas = htmlspecialchars($data[3] ?? '');
                $ruangan = htmlspecialchars($data[4] ?? '');
                $wa = htmlspecialchars($data[5] ?? '');
                $foto_default = 'default.png'; 

                if (!empty($id_siswa) && !empty($nama)) {
                    $qr_content = preg_replace('/^QR/i', '', $qr_content);
                    if(empty($qr_content)) $qr_content = $id_siswa;
                    
                    $check = $conn->prepare("SELECT id FROM students WHERE id = ?");
                    $check->bind_param("s", $id_siswa);
                    $check->execute(); $check->store_result();
                    if ($check->num_rows == 0) {
                        $stmt = $conn->prepare("INSERT INTO students (id, qr_code_content, nama, kelas, ruangan, foto, nomor_wa) VALUES (?, ?, ?, ?, ?, ?, ?)");
                        $stmt->bind_param("sssssss", $id_siswa, $qr_content, $nama, $kelas, $ruangan, $foto_default, $wa);
                        if ($stmt->execute()) { $success_count++; }
                        $stmt->close();
                    }
                    $check->close();
                }
            }
            fclose($handle);
            $message = "Berhasil mengimpor $success_count data siswa.";
            $message_type = "success";
        } else {
            $message = "Format file salah. Harap upload file .csv";
            $message_type = "error";
        }
    }

    // ... (Logika Edit Siswa) ...
    if (isset($_POST['edit_siswa'])) {
        $id = $_POST['edit_id'];
        $nama = htmlspecialchars($_POST['edit_nama']);
        $kelas = htmlspecialchars($_POST['edit_kelas']);
        $ruangan = htmlspecialchars($_POST['edit_ruangan']);
        $wa = htmlspecialchars($_POST['edit_wa']);
        
        $upload_ok = true; $foto_sql_part = ""; $params = [$nama, $kelas, $ruangan, $wa]; $types = "ssss";

        if (isset($_FILES['edit_foto']) && $_FILES['edit_foto']['error'] === UPLOAD_ERR_OK) {
            $target_dir = "uploads/";
            if (!is_dir($target_dir)) mkdir($target_dir, 0755, true);
            $file_ext = strtolower(pathinfo($_FILES['edit_foto']['name'], PATHINFO_EXTENSION));
            $allowed_ext = ['jpg', 'jpeg', 'png', 'gif'];
            if (in_array($file_ext, $allowed_ext)) {
                $new_filename = "foto_" . $id . "_" . time() . "." . $file_ext;
                $target_file = $target_dir . $new_filename;
                if (move_uploaded_file($_FILES['edit_foto']['tmp_name'], $target_file)) {
                    $foto_sql_part = ", foto = ?"; $params[] = $new_filename; $types .= "s";
                } else {
                    $message = "Gagal mengupload foto."; $message_type = "error"; $upload_ok = false;
                }
            } else {
                $message = "Format foto salah."; $message_type = "error"; $upload_ok = false;
            }
        }
        if ($upload_ok) {
            $params[] = $id; $types .= "s";
            $stmt = $conn->prepare("UPDATE students SET nama = ?, kelas = ?, ruangan = ?, nomor_wa = ?" . $foto_sql_part . " WHERE id = ?");
            $stmt->bind_param($types, ...$params);
            if ($stmt->execute()) { $message = "Data diperbarui."; $message_type = "success"; }
            else { $message = "Gagal update DB."; $message_type = "error"; }
            $stmt->close();
        }
    }

    // ... (Logika Hapus Terpilih) ...
    if (isset($_POST['hapus_terpilih'])) {
        if (!empty($_POST['ids']) && is_array($_POST['ids'])) {
            $ids = $_POST['ids']; $count = 0;
            $stmt = $conn->prepare("DELETE FROM students WHERE id = ?");
            foreach ($ids as $id_del) { $stmt->bind_param("s", $id_del); if ($stmt->execute()) $count++; }
            $stmt->close();
            $message = $count > 0 ? "Berhasil menghapus $count data." : "Gagal menghapus.";
            $message_type = $count > 0 ? "success" : "error";
        } else { $message = "Tidak ada data dipilih."; $message_type = "error"; }
    }

    // ... (Logika Reset Sesi Terpilih / Bulk Reset) ...
    if (isset($_POST['reset_sesi_terpilih'])) {
        if (!empty($_POST['ids']) && is_array($_POST['ids'])) {
            $ids = $_POST['ids']; $count = 0;
            $stmt = $conn->prepare("UPDATE students SET last_logged_in_device_id = NULL, last_activity = NULL WHERE id = ?");
            foreach ($ids as $id_reset) { 
                $stmt->bind_param("s", $id_reset); 
                if ($stmt->execute()) $count++; 
            }
            $stmt->close();
            $message = $count > 0 ? "Berhasil mereset sesi $count siswa." : "Gagal mereset sesi.";
            $message_type = $count > 0 ? "success" : "error";
        } else { $message = "Tidak ada siswa yang dipilih."; $message_type = "error"; }
    }

    // ... (Logika Reset Sesi Single - Tombol di baris tabel) ...
    if (isset($_POST['reset_session'])) {
        $id_reset = $_POST['reset_id'];
        $stmt = $conn->prepare("UPDATE students SET last_logged_in_device_id = NULL, last_activity = NULL WHERE id = ?");
        $stmt->bind_param("s", $id_reset);
        if ($stmt->execute()) { $message = "Sesi direset."; $message_type = "success"; }
        else { $message = "Gagal reset."; $message_type = "error"; }
        $stmt->close();
    }
}

// 3. LOGIKA DOWNLOAD TEMPLATE (GET)
if (isset($_GET['action']) && $_GET['action'] == 'download_template') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="template_siswa_baru.csv"');
    $output = fopen('php://output', 'w');
    fputcsv($output, array('ID Siswa (Unik)', 'Isi QR Code', 'Nama Lengkap', 'Kelas', 'Ruangan', 'Nomor WA'));
    fputcsv($output, array('12345', '12345', 'Ahmad Contoh', 'XH1', 'R.01', '08123456789'));
    fclose($output); exit;
}

// 4. LOGIKA PENCARIAN & FILTER (Digunakan untuk Halaman Utama & AJAX)
$search_term = $_GET['search'] ?? '';
$filter_class = $_GET['kelas_filter'] ?? '';
$filter_ruangan = $_GET['ruangan_filter'] ?? '';
$filter_status = $_GET['status_filter'] ?? ''; 

$sql = "SELECT * FROM students WHERE 1=1";
$params = [];
$types = "";

if (!empty($search_term)) {
    $sql .= " AND (nama LIKE ? OR id LIKE ? OR qr_code_content LIKE ?)";
    $term = "%" . $search_term . "%";
    $params[] = $term; $params[] = $term; $params[] = $term;
    $types .= "sss";
}
if (!empty($filter_class)) {
    $sql .= " AND kelas = ?";
    $params[] = $filter_class;
    $types .= "s";
}
if (!empty($filter_ruangan)) {
    $sql .= " AND ruangan = ?";
    $params[] = $filter_ruangan;
    $types .= "s";
}
// Logika Filter Status SQL
if ($filter_status === 'online') {
    $sql .= " AND last_activity >= DATE_SUB(NOW(), INTERVAL 5 MINUTE)";
} elseif ($filter_status === 'offline') {
    $sql .= " AND (last_activity < DATE_SUB(NOW(), INTERVAL 5 MINUTE) OR last_activity IS NULL)";
}

$sql .= " ORDER BY last_activity DESC, kelas ASC, nama ASC";

$stmt = $conn->prepare($sql);
if (!empty($params)) { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$result = $stmt->get_result();
$students = $result->fetch_all(MYSQLI_ASSOC);

// --- BLOCK AJAX HANDLER ---
if (isset($_GET['ajax_request'])) {
    if (empty($students)) {
        echo '<tr><td colspan="7" class="p-10 text-center flex flex-col items-center justify-center text-slate-500">
                <i data-lucide="inbox" class="w-10 h-10 mb-2 text-slate-300"></i><p>Data tidak ditemukan.</p></td></tr>';
    } else {
        foreach ($students as $siswa) {
            $statusData = getTimeStatus($siswa['last_activity'] ?? null);
            $statusClass = $statusData['status'] == 'online' ? 'bg-green-100 text-green-700 border-green-200' : 'bg-slate-100 text-slate-500 border-slate-200';
            $dotClass = $statusData['status'] == 'online' ? 'bg-green-500' : 'bg-slate-400';
            ?>
            <tr class="hover:bg-blue-50/50 transition-colors group">
                <td class="p-4 text-center">
                    <input type="checkbox" name="ids[]" value="<?php echo $siswa['id']; ?>" class="student-checkbox rounded border-slate-300 text-blue-600 focus:ring-blue-500 cursor-pointer">
                </td>
                <td class="p-4">
                    <div class="font-mono text-xs text-slate-500 font-bold"><?php echo htmlspecialchars($siswa['id']); ?></div>
                    <div class="text-xs text-slate-400 mt-0.5" title="QR Content"><i data-lucide="qr-code" class="w-3 h-3 inline mr-1"></i><?php echo htmlspecialchars($siswa['qr_code_content']); ?></div>
                </td>
                <td class="p-4 font-medium text-slate-900 group-hover:text-blue-700">
                    <?php echo htmlspecialchars($siswa['nama']); ?>
                </td>
                <td class="p-4">
                    <div class="flex flex-col gap-1">
                        <span class="inline-flex w-fit items-center px-2 py-0.5 rounded text-xs font-medium bg-blue-50 text-blue-700 border border-blue-100">
                            <?php echo htmlspecialchars($siswa['kelas']); ?>
                        </span>
                        <span class="inline-flex w-fit items-center px-2 py-0.5 rounded text-xs font-medium bg-amber-50 text-amber-700 border border-amber-100">
                            <i data-lucide="map-pin" class="w-3 h-3 mr-1"></i>
                            <?php echo htmlspecialchars($siswa['ruangan']); ?>
                        </span>
                    </div>
                </td>
                <td class="p-4">
                    <div class="inline-flex items-center gap-2 px-2.5 py-1 rounded-full text-xs font-medium border <?php echo $statusClass; ?>">
                        <span class="relative flex h-2 w-2">
                            <?php if($statusData['status'] == 'online'): ?>
                            <span class="animate-ping absolute inline-flex h-full w-full rounded-full bg-green-400 opacity-75"></span>
                            <?php endif; ?>
                            <span class="relative inline-flex rounded-full h-2 w-2 <?php echo $dotClass; ?>"></span>
                        </span>
                        <?php echo $statusData['text']; ?>
                    </div>
                    <div class="text-[10px] text-slate-400 mt-1 ml-1">
                        <?php echo $statusData['time'] ? $statusData['time'] : '-'; ?>
                    </div>
                </td>
                <td class="p-4 text-slate-500 text-xs">
                    <div class="flex items-center gap-1"><i data-lucide="phone" class="w-3 h-3"></i> <?php echo htmlspecialchars($siswa['nomor_wa']); ?></div>
                </td>
                <td class="p-4 text-center">
                    <div class="flex items-center justify-center gap-2">
                        <button type="button" onclick="showQrModal('<?php echo htmlspecialchars($siswa['qr_code_content']); ?>', '<?php echo htmlspecialchars($siswa['nama'], ENT_QUOTES); ?>')" class="inline-flex items-center justify-center p-1.5 text-slate-500 hover:text-blue-600 bg-white border border-slate-300 rounded-lg hover:bg-blue-50 transition" title="Lihat QR">
                            <i data-lucide="qr-code" class="w-4 h-4"></i>
                        </button>
                        <button type="button" onclick="openEditModal('<?php echo $siswa['id']; ?>','<?php echo htmlspecialchars($siswa['nama'], ENT_QUOTES); ?>','<?php echo htmlspecialchars($siswa['kelas'], ENT_QUOTES); ?>','<?php echo htmlspecialchars($siswa['ruangan'], ENT_QUOTES); ?>','<?php echo htmlspecialchars($siswa['nomor_wa'], ENT_QUOTES); ?>')" class="inline-flex items-center justify-center p-1.5 text-slate-500 hover:text-amber-600 bg-white border border-slate-300 rounded-lg hover:bg-amber-50 transition" title="Edit Data & Foto">
                            <i data-lucide="edit-3" class="w-4 h-4"></i>
                        </button>
                        <button type="button" onclick="confirmResetSession('<?php echo $siswa['id']; ?>', '<?php echo htmlspecialchars($siswa['nama'], ENT_QUOTES); ?>')" class="inline-flex items-center justify-center p-1.5 text-slate-500 hover:text-red-600 bg-white border border-slate-300 rounded-lg hover:bg-red-50 transition" title="Hapus Sesi Login">
                            <i data-lucide="log-out" class="w-4 h-4"></i>
                        </button>
                    </div>
                </td>
            </tr>
            <?php
        }
    }
    // STOP EKSEKUSI AGAR TIDAK MERENDER HALAMAN FULL
    exit;
}
// --- END AJAX BLOCK ---

// Dropdown Data (Untuk Form Filter Awal)
$class_query = "SELECT DISTINCT kelas FROM students ORDER BY kelas ASC";
$class_result = $conn->query($class_query);
$available_classes = [];
while ($row = $class_result->fetch_assoc()) { if(!empty($row['kelas'])) $available_classes[] = $row['kelas']; }

$room_query = "SELECT DISTINCT ruangan FROM students ORDER BY ruangan ASC";
$room_result = $conn->query($room_query);
$available_rooms = [];
while ($row = $room_result->fetch_assoc()) { if(!empty($row['ruangan'])) $available_rooms[] = $row['ruangan']; }
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manajemen Siswa - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://unpkg.com/lucide@latest"></script>
    <style> body { font-family: 'Plus Jakarta Sans', sans-serif; } </style>
</head>
<body class="bg-slate-50 text-slate-800 min-h-screen flex">

    <!-- Sidebar -->
    <aside class="w-64 bg-white border-r border-slate-200 hidden md:flex flex-col fixed h-full z-10">
        <div class="p-6 border-b border-slate-100 flex items-center gap-3">
            <div class="w-8 h-8 bg-blue-600 rounded-lg flex items-center justify-center text-white font-bold text-lg">A</div>
            <h1 class="text-lg font-bold text-slate-800">Admin Panel</h1>
        </div>
        <nav class="flex-1 p-4 space-y-1 overflow-y-auto">
            <p class="px-4 text-xs font-semibold text-slate-400 uppercase tracking-wider mb-2">Menu Utama</p>
            <a href="daftarsiswa.php" class="flex items-center gap-3 px-4 py-3 bg-blue-50 text-blue-700 rounded-lg text-sm font-medium transition-colors">
                <i data-lucide="users" class="w-5 h-5"></i> Data Siswa
            </a>
            <a href="cetakkartu.php" class="flex items-center gap-3 px-4 py-3 text-slate-600 hover:bg-slate-50 hover:text-slate-900 rounded-lg text-sm font-medium transition-colors">
                <i data-lucide="printer" class="w-5 h-5"></i> Cetak Kartu
            </a>
            <div class="pt-4 mt-4 border-t border-slate-100">
                <a href="logout.php" class="flex items-center gap-3 px-4 py-3 text-red-600 hover:bg-red-50 rounded-lg text-sm font-medium transition-colors">
                    <i data-lucide="log-out" class="w-5 h-5"></i> Logout
                </a>
            </div>
        </nav>
        <div class="p-4 bg-slate-50 border-t border-slate-200">
            <div class="flex items-center gap-3">
                <div class="w-8 h-8 rounded-full bg-blue-100 text-blue-600 flex items-center justify-center text-xs font-bold"><?php echo substr($nama_pengguna, 0, 2); ?></div>
                <div><p class="text-sm font-medium text-slate-700"><?php echo $nama_pengguna; ?></p></div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="flex-1 md:ml-64 p-4 md:p-8">
        
        <!-- Header -->
        <div class="flex flex-col md:flex-row md:items-center justify-between gap-4 mb-8">
            <div>
                <h2 class="text-2xl font-bold text-slate-900">Daftar Siswa</h2>
                <p class="text-slate-500 text-sm mt-1">Kelola data siswa, ruangan, status online, dan cetak kartu.</p>
            </div>
            
            <div class="flex items-center gap-3">
                <a href="?action=download_template" class="flex items-center gap-2 bg-white border border-slate-300 text-slate-700 px-4 py-2 rounded-lg text-sm font-medium hover:bg-slate-50 transition shadow-sm">
                    <i data-lucide="download" class="w-4 h-4"></i> Template
                </a>
                <button onclick="toggleModal('importModal')" class="flex items-center gap-2 bg-green-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-green-700 transition shadow-sm">
                    <i data-lucide="upload" class="w-4 h-4"></i> Import Excel
                </button>
            </div>
        </div>

        <?php if (!empty($message)): ?>
            <div class="mb-6 p-4 rounded-lg flex items-center gap-3 <?php echo $message_type === 'success' ? 'bg-green-50 text-green-700 border border-green-200' : 'bg-red-50 text-red-700 border border-red-200'; ?>">
                <i data-lucide="<?php echo $message_type === 'success' ? 'check-circle' : 'alert-circle'; ?>" class="w-5 h-5"></i>
                <span class="text-sm font-medium"><?php echo $message; ?></span>
            </div>
        <?php endif; ?>

        <!-- Toolbar Filter Realtime -->
        <div class="bg-white p-4 rounded-xl shadow-sm border border-slate-200 mb-6">
            <div class="flex flex-col md:flex-row gap-4 justify-between items-center">
                <div class="w-full md:w-auto flex flex-col md:flex-row items-center gap-3">
                    <!-- Filter Kelas -->
                    <div class="relative w-full md:w-40">
                        <i data-lucide="filter" class="absolute left-3 top-2.5 w-4 h-4 text-slate-400"></i>
                        <select id="kelasFilter" class="w-full pl-9 pr-4 py-2 bg-slate-50 border border-slate-300 text-slate-700 text-sm rounded-lg focus:ring-2 focus:ring-blue-500 outline-none cursor-pointer">
                            <option value="">Semua Kelas</option>
                            <?php foreach ($available_classes as $kls): ?>
                                <option value="<?php echo htmlspecialchars($kls); ?>" <?php echo ($filter_class == $kls) ? 'selected' : ''; ?>><?php echo htmlspecialchars($kls); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <!-- Filter Ruangan -->
                    <div class="relative w-full md:w-40">
                        <i data-lucide="map-pin" class="absolute left-3 top-2.5 w-4 h-4 text-slate-400"></i>
                        <select id="ruanganFilter" class="w-full pl-9 pr-4 py-2 bg-slate-50 border border-slate-300 text-slate-700 text-sm rounded-lg focus:ring-2 focus:ring-blue-500 outline-none cursor-pointer">
                            <option value="">Semua Ruangan</option>
                            <?php foreach ($available_rooms as $room): ?>
                                <option value="<?php echo htmlspecialchars($room); ?>" <?php echo ($filter_ruangan == $room) ? 'selected' : ''; ?>><?php echo htmlspecialchars($room); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <!-- Filter Status (BARU) -->
                    <div class="relative w-full md:w-40">
                        <i data-lucide="activity" class="absolute left-3 top-2.5 w-4 h-4 text-slate-400"></i>
                        <select id="statusFilter" class="w-full pl-9 pr-4 py-2 bg-slate-50 border border-slate-300 text-slate-700 text-sm rounded-lg focus:ring-2 focus:ring-blue-500 outline-none cursor-pointer">
                            <option value="">Semua Status</option>
                            <option value="online" <?php echo ($filter_status == 'online') ? 'selected' : ''; ?>>Online</option>
                            <option value="offline" <?php echo ($filter_status == 'offline') ? 'selected' : ''; ?>>Offline</option>
                        </select>
                    </div>
                </div>
                <div class="w-full md:w-auto flex items-center gap-2">
                    <!-- Search Input -->
                    <div class="relative w-full md:w-64">
                        <i data-lucide="search" class="absolute left-3 top-2.5 w-4 h-4 text-slate-400"></i>
                        <input type="text" id="searchInput" placeholder="Cari ID, Nama, QR..." value="<?php echo htmlspecialchars($search_term); ?>" class="w-full pl-9 pr-4 py-2 bg-slate-50 border border-slate-300 text-slate-700 text-sm rounded-lg focus:ring-2 focus:ring-blue-500 outline-none">
                    </div>
                    <!-- Indikator Loading -->
                    <div id="loadingIndicator" class="hidden text-blue-600 animate-spin">
                        <i data-lucide="loader-2" class="w-5 h-5"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Tabel Data -->
        <form action="" method="POST" id="bulkActionForm">
            <div class="flex flex-col md:flex-row justify-between items-center mb-4 px-1 gap-3">
                <p class="text-sm text-slate-500">Menampilkan hasil <span id="dataCount" class="font-bold"><?php echo count($students); ?></span> siswa</p>
                <div class="flex gap-2">
                    <button type="submit" name="hapus_terpilih" onclick="confirmBulkAction(event, 'hapus_terpilih', 'Hapus Data Terpilih', 'Yakin ingin menghapus semua data siswa yang dipilih? Tindakan ini tidak dapat dibatalkan.', 'danger')" class="flex items-center gap-2 bg-red-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-red-700 transition shadow-sm">
                        <i data-lucide="trash-2" class="w-4 h-4"></i> Hapus Terpilih
                    </button>
                    <!-- Tombol Baru: Reset Sesi Terpilih -->
                    <button type="submit" name="reset_sesi_terpilih" onclick="confirmBulkAction(event, 'reset_sesi_terpilih', 'Reset Sesi Terpilih', 'Yakin ingin mereset sesi login siswa yang dipilih? Siswa harus login ulang.', 'warning')" class="flex items-center gap-2 bg-amber-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-amber-700 transition shadow-sm">
                        <i data-lucide="refresh-cw" class="w-4 h-4"></i> Reset Sesi Terpilih
                    </button>
                    <button type="submit" formaction="cetakkartu.php" formtarget="_blank" class="flex items-center gap-2 bg-indigo-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-indigo-700 transition shadow-sm">
                        <i data-lucide="printer" class="w-4 h-4"></i> Cetak Kartu
                    </button>
                </div>
            </div>

            <div class="bg-white rounded-xl shadow-sm border border-slate-200 overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="w-full text-left border-collapse">
                        <thead>
                            <tr class="bg-slate-50 border-b border-slate-200 text-xs uppercase text-slate-500 font-semibold tracking-wider">
                                <th class="p-4 w-10 text-center"><input type="checkbox" id="selectAll" class="rounded border-slate-300 text-blue-600 focus:ring-blue-500"></th>
                                <th class="p-4">ID / QR Content</th>
                                <th class="p-4">Nama Lengkap</th>
                                <th class="p-4">Kelas / Ruangan</th>
                                <th class="p-4">Status</th>
                                <th class="p-4">Info Kontak</th>
                                <th class="p-4 text-center">Aksi</th>
                            </tr>
                        </thead>
                        <tbody id="studentTableBody" class="divide-y divide-slate-100 text-sm text-slate-700">
                            <!-- ISI TABEL AWAL (Akan direplace oleh AJAX) -->
                            <?php
                                // Kita gunakan loop yang sama seperti di blok AJAX
                                // Namun untuk load pertama kali, kita tulis manual/include.
                                // Untuk mempersingkat, kode di bawah SAMA PERSIS dengan blok AJAX
                                if (empty($students)) {
                                    echo '<tr><td colspan="7" class="p-10 text-center flex flex-col items-center justify-center text-slate-500">
                                            <i data-lucide="inbox" class="w-10 h-10 mb-2 text-slate-300"></i><p>Data tidak ditemukan.</p></td></tr>';
                                } else {
                                    foreach ($students as $siswa) {
                                        $statusData = getTimeStatus($siswa['last_activity'] ?? null);
                                        $statusClass = $statusData['status'] == 'online' ? 'bg-green-100 text-green-700 border-green-200' : 'bg-slate-100 text-slate-500 border-slate-200';
                                        $dotClass = $statusData['status'] == 'online' ? 'bg-green-500' : 'bg-slate-400';
                                        ?>
                                        <tr class="hover:bg-blue-50/50 transition-colors group">
                                            <td class="p-4 text-center"><input type="checkbox" name="ids[]" value="<?php echo $siswa['id']; ?>" class="student-checkbox rounded border-slate-300 text-blue-600 focus:ring-blue-500 cursor-pointer"></td>
                                            <td class="p-4">
                                                <div class="font-mono text-xs text-slate-500 font-bold"><?php echo htmlspecialchars($siswa['id']); ?></div>
                                                <div class="text-xs text-slate-400 mt-0.5"><i data-lucide="qr-code" class="w-3 h-3 inline mr-1"></i><?php echo htmlspecialchars($siswa['qr_code_content']); ?></div>
                                            </td>
                                            <td class="p-4 font-medium text-slate-900 group-hover:text-blue-700"><?php echo htmlspecialchars($siswa['nama']); ?></td>
                                            <td class="p-4">
                                                <div class="flex flex-col gap-1">
                                                    <span class="inline-flex w-fit items-center px-2 py-0.5 rounded text-xs font-medium bg-blue-50 text-blue-700 border border-blue-100"><?php echo htmlspecialchars($siswa['kelas']); ?></span>
                                                    <span class="inline-flex w-fit items-center px-2 py-0.5 rounded text-xs font-medium bg-amber-50 text-amber-700 border border-amber-100"><i data-lucide="map-pin" class="w-3 h-3 mr-1"></i><?php echo htmlspecialchars($siswa['ruangan']); ?></span>
                                                </div>
                                            </td>
                                            <td class="p-4">
                                                <div class="inline-flex items-center gap-2 px-2.5 py-1 rounded-full text-xs font-medium border <?php echo $statusClass; ?>">
                                                    <span class="relative flex h-2 w-2">
                                                        <?php if($statusData['status'] == 'online'): ?><span class="animate-ping absolute inline-flex h-full w-full rounded-full bg-green-400 opacity-75"></span><?php endif; ?>
                                                        <span class="relative inline-flex rounded-full h-2 w-2 <?php echo $dotClass; ?>"></span>
                                                    </span>
                                                    <?php echo $statusData['text']; ?>
                                                </div>
                                                <div class="text-[10px] text-slate-400 mt-1 ml-1"><?php echo $statusData['time'] ? $statusData['time'] : '-'; ?></div>
                                            </td>
                                            <td class="p-4 text-slate-500 text-xs"><div class="flex items-center gap-1"><i data-lucide="phone" class="w-3 h-3"></i> <?php echo htmlspecialchars($siswa['nomor_wa']); ?></div></td>
                                            <td class="p-4 text-center">
                                                <div class="flex items-center justify-center gap-2">
                                                    <button type="button" onclick="showQrModal('<?php echo htmlspecialchars($siswa['qr_code_content']); ?>', '<?php echo htmlspecialchars($siswa['nama'], ENT_QUOTES); ?>')" class="inline-flex items-center justify-center p-1.5 text-slate-500 hover:text-blue-600 bg-white border border-slate-300 rounded-lg hover:bg-blue-50 transition"><i data-lucide="qr-code" class="w-4 h-4"></i></button>
                                                    <button type="button" onclick="openEditModal('<?php echo $siswa['id']; ?>','<?php echo htmlspecialchars($siswa['nama'], ENT_QUOTES); ?>','<?php echo htmlspecialchars($siswa['kelas'], ENT_QUOTES); ?>','<?php echo htmlspecialchars($siswa['ruangan'], ENT_QUOTES); ?>','<?php echo htmlspecialchars($siswa['nomor_wa'], ENT_QUOTES); ?>')" class="inline-flex items-center justify-center p-1.5 text-slate-500 hover:text-amber-600 bg-white border border-slate-300 rounded-lg hover:bg-amber-50 transition"><i data-lucide="edit-3" class="w-4 h-4"></i></button>
                                                    <button type="button" onclick="confirmResetSession('<?php echo $siswa['id']; ?>', '<?php echo htmlspecialchars($siswa['nama'], ENT_QUOTES); ?>')" class="inline-flex items-center justify-center p-1.5 text-slate-500 hover:text-red-600 bg-white border border-slate-300 rounded-lg hover:bg-red-50 transition"><i data-lucide="log-out" class="w-4 h-4"></i></button>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php
                                    }
                                }
                            ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </form>
    </main>

    <!-- FORM TERSEMBUNYI, MODALS (SAMA SEPERTI SEBELUMNYA) -->
    <form id="resetSessionForm" action="" method="POST" style="display:none;"><input type="hidden" name="reset_session" value="1"><input type="hidden" name="reset_id" id="reset_session_id"></form>
    
    <!-- Modal Import -->
    <div id="importModal" class="fixed inset-0 z-50 hidden">
        <div class="absolute inset-0 bg-slate-900/50 backdrop-blur-sm" onclick="toggleModal('importModal')"></div>
        <div class="relative min-h-screen flex items-center justify-center p-4">
            <div class="bg-white rounded-xl shadow-2xl w-full max-w-md p-6">
                <div class="flex justify-between items-center mb-6"><h3 class="text-lg font-bold">Import Data</h3><button onclick="toggleModal('importModal')"><i data-lucide="x" class="w-5 h-5"></i></button></div>
                <form action="" method="POST" enctype="multipart/form-data">
                    <div class="bg-blue-50 p-4 mb-4 text-xs text-blue-700 rounded">Format: 1.ID, 2.QR, 3.Nama, 4.Kelas, 5.Ruangan, 6.WA</div>
                    <input type="file" name="file_csv" accept=".csv" required class="block w-full text-sm text-slate-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:bg-blue-50 file:text-blue-700 border rounded-lg">
                    <div class="pt-4 flex gap-3"><button type="button" onclick="toggleModal('importModal')" class="flex-1 py-2 border rounded">Batal</button><button type="submit" name="import_data" class="flex-1 py-2 bg-blue-600 text-white rounded">Import</button></div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Modal Edit -->
    <div id="editModal" class="fixed inset-0 z-50 hidden">
        <div class="absolute inset-0 bg-slate-900/50 backdrop-blur-sm" onclick="closeEditModal()"></div>
        <div class="relative min-h-screen flex items-center justify-center p-4">
            <div class="bg-white rounded-xl shadow-2xl w-full max-w-lg p-6">
                <div class="flex justify-between mb-6"><h3 class="text-lg font-bold">Edit Siswa</h3><button onclick="closeEditModal()"><i data-lucide="x" class="w-5 h-5"></i></button></div>
                <form action="" method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="edit_siswa" value="1"><input type="hidden" name="edit_id" id="edit_id">
                    <div class="grid grid-cols-2 gap-4">
                        <div class="col-span-2"><label class="text-sm font-medium">Nama</label><input type="text" name="edit_nama" id="edit_nama" required class="w-full px-3 py-2 border rounded-lg"></div>
                        <div><label class="text-sm font-medium">Kelas</label><input type="text" name="edit_kelas" id="edit_kelas" required class="w-full px-3 py-2 border rounded-lg"></div>
                        <div><label class="text-sm font-medium">Ruangan</label><input type="text" name="edit_ruangan" id="edit_ruangan" required class="w-full px-3 py-2 border rounded-lg"></div>
                        <div class="col-span-2"><label class="text-sm font-medium">WhatsApp</label><input type="text" name="edit_wa" id="edit_wa" class="w-full px-3 py-2 border rounded-lg"></div>
                        <div class="col-span-2"><label class="text-sm font-medium">Foto</label><input type="file" name="edit_foto" class="block w-full text-sm border rounded-lg"></div>
                    </div>
                    <div class="pt-6 flex gap-3"><button type="button" onclick="closeEditModal()" class="flex-1 py-2 border rounded">Batal</button><button type="submit" class="flex-1 py-2 bg-amber-500 text-white rounded">Simpan</button></div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal QR -->
    <div id="qrModal" class="fixed inset-0 z-50 hidden">
        <div class="absolute inset-0 bg-slate-900/50 backdrop-blur-sm" onclick="closeQrModal()"></div>
        <div class="relative min-h-screen flex items-center justify-center p-4">
            <div class="bg-white rounded-xl shadow-2xl w-full max-w-sm p-6 text-center">
                <button onclick="closeQrModal()" class="absolute top-4 right-4"><i data-lucide="x" class="w-5 h-5"></i></button>
                <h3 class="text-lg font-bold mb-1" id="qrModalName">Nama</h3>
                <div class="bg-white p-4 border rounded-xl inline-block mb-4"><img id="qrModalImage" src="" class="w-48 h-48 object-contain"></div>
                <div class="flex justify-center mb-6"><div class="bg-slate-100 py-1 px-4 rounded-full text-xs font-mono" id="qrModalContent">CODE</div></div>
            </div>
        </div>
    </div>

    <!-- MODAL KONFIRMASI (BARU) -->
    <div id="confirmModal" class="fixed inset-0 z-50 hidden">
        <div class="absolute inset-0 bg-slate-900/50 backdrop-blur-sm transition-opacity" onclick="closeConfirmModal()"></div>
        <div class="relative min-h-screen flex items-center justify-center p-4">
            <div class="bg-white rounded-xl shadow-2xl w-full max-w-sm p-6 transform transition-all scale-100">
                <div class="flex items-center gap-3 mb-4">
                    <div class="w-10 h-10 rounded-full bg-amber-100 flex items-center justify-center text-amber-600" id="confirmIcon">
                        <i data-lucide="alert-triangle" class="w-6 h-6"></i>
                    </div>
                    <h3 class="text-lg font-bold text-slate-800" id="confirmTitle">Konfirmasi</h3>
                </div>
                <p class="text-sm text-slate-600 mb-6" id="confirmMessage">Apakah Anda yakin?</p>
                <div class="flex gap-3">
                    <button onclick="closeConfirmModal()" class="flex-1 px-4 py-2 border border-slate-300 text-slate-700 rounded-lg text-sm font-medium hover:bg-slate-50">Batal</button>
                    <button id="confirmActionBtn" class="flex-1 px-4 py-2 bg-red-600 text-white rounded-lg text-sm font-medium hover:bg-red-700">Ya, Lanjutkan</button>
                </div>
            </div>
        </div>
    </div>

    <!-- SCRIPTS UNTUK PENCARIAN REALTIME & MODAL -->
    <script>
        lucide.createIcons();

        // 1. Logic Realtime Search & Filter AJAX
        const searchInput = document.getElementById('searchInput');
        const kelasFilter = document.getElementById('kelasFilter');
        const ruanganFilter = document.getElementById('ruanganFilter');
        const statusFilter = document.getElementById('statusFilter'); // Filter Baru
        const tableBody = document.getElementById('studentTableBody');
        const loadingIndicator = document.getElementById('loadingIndicator');
        const dataCount = document.getElementById('dataCount');

        let debounceTimeout = null;

        function loadData() {
            // Tampilkan loading
            loadingIndicator.classList.remove('hidden');

            const params = new URLSearchParams({
                ajax_request: '1', // Sinyal ke PHP ini AJAX
                search: searchInput.value,
                kelas_filter: kelasFilter.value,
                ruangan_filter: ruanganFilter.value,
                status_filter: statusFilter.value
            });

            fetch('daftarsiswa.php?' + params.toString())
                .then(response => response.text())
                .then(html => {
                    tableBody.innerHTML = html;
                    lucide.createIcons(); // Re-init icons pada konten baru
                    loadingIndicator.classList.add('hidden');
                    
                    // Update jumlah data (opsional: parsing HTML untuk hitung tr atau minta JSON)
                    // Disini kita hitung row tabel
                    const rowCount = tableBody.querySelectorAll('tr').length;
                    dataCount.innerText = rowCount;
                })
                .catch(err => {
                    console.error('Error fetching data:', err);
                    loadingIndicator.classList.add('hidden');
                });
        }

        // Event Listeners
        searchInput.addEventListener('input', () => {
            clearTimeout(debounceTimeout);
            debounceTimeout = setTimeout(loadData, 300); // Debounce biar gak spam request
        });
        kelasFilter.addEventListener('change', loadData);
        ruanganFilter.addEventListener('change', loadData);
        statusFilter.addEventListener('change', loadData);

        // 2. Modal Functions (Sama seperti sebelumnya)
        function toggleModal(id) { document.getElementById(id).classList.toggle('hidden'); }
        function showQrModal(content, name) {
            document.getElementById('qrModalName').innerText = name;
            document.getElementById('qrModalContent').innerText = content;
            document.getElementById('qrModalImage').src = `https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=${content}`;
            toggleModal('qrModal');
        }
        function closeQrModal() { toggleModal('qrModal'); }
        function openEditModal(id, nama, kelas, ruangan, wa) {
            document.getElementById('edit_id').value = id;
            document.getElementById('edit_nama').value = nama;
            document.getElementById('edit_kelas').value = kelas;
            document.getElementById('edit_ruangan').value = ruangan;
            document.getElementById('edit_wa').value = wa;
            toggleModal('editModal');
        }
        function closeEditModal() { toggleModal('editModal'); }

        // --- KONFIRMASI MODAL LOGIC (BARU) ---
        let pendingAction = null;

        function showConfirmModal(title, message, action, type = 'danger') {
            document.getElementById('confirmTitle').innerText = title;
            document.getElementById('confirmMessage').innerText = message;
            
            const btn = document.getElementById('confirmActionBtn');
            const iconContainer = document.getElementById('confirmIcon');
            
            if (type === 'danger') {
                btn.className = "flex-1 px-4 py-2 bg-red-600 text-white rounded-lg text-sm font-medium hover:bg-red-700";
                iconContainer.className = "w-10 h-10 rounded-full bg-red-100 flex items-center justify-center text-red-600";
                iconContainer.innerHTML = '<i data-lucide="alert-triangle" class="w-6 h-6"></i>';
            } else {
                btn.className = "flex-1 px-4 py-2 bg-amber-600 text-white rounded-lg text-sm font-medium hover:bg-amber-700";
                iconContainer.className = "w-10 h-10 rounded-full bg-amber-100 flex items-center justify-center text-amber-600";
                iconContainer.innerHTML = '<i data-lucide="help-circle" class="w-6 h-6"></i>';
            }

            pendingAction = action;
            toggleModal('confirmModal');
            lucide.createIcons(); // Re-render icon
        }

        function closeConfirmModal() {
            toggleModal('confirmModal');
            pendingAction = null;
        }

        document.getElementById('confirmActionBtn').addEventListener('click', function() {
            if (pendingAction) pendingAction();
            closeConfirmModal();
        });

        // Wrapper untuk reset per siswa
        function confirmResetSession(id, name) {
            showConfirmModal(
                'Reset Sesi Login?', 
                `Apakah Anda yakin ingin mereset sesi untuk siswa "${name}"? Siswa akan logout otomatis.`, 
                () => {
                    document.getElementById('reset_session_id').value = id;
                    document.getElementById('resetSessionForm').submit();
                },
                'warning'
            );
        }

        // Wrapper untuk aksi massal (Bulk)
        function confirmBulkAction(event, actionName, title, message, type) {
            event.preventDefault();
            const form = document.getElementById('bulkActionForm');
            
            // Cek apakah ada checkbox yang dipilih
            const checkboxes = form.querySelectorAll('input[name="ids[]"]:checked');
            if (checkboxes.length === 0) {
                // Bisa pakai alert biasa atau modal info, kita pakai alert biasa dulu agar simple
                alert("Silakan pilih minimal satu data siswa.");
                return;
            }

            showConfirmModal(title, message, () => {
                // Buat hidden input untuk nama tombol (karena submit via JS tidak kirim name tombol)
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = actionName;
                input.value = '1';
                form.appendChild(input);
                form.submit();
            }, type);
        }
        // -------------------------------------

        document.getElementById('selectAll').addEventListener('change', function(e) {
            // Gunakan querySelectorAll pada saat event fired agar mengambil checkbox terbaru (hasil AJAX)
            const checkboxes = document.querySelectorAll('.student-checkbox');
            checkboxes.forEach(cb => cb.checked = e.target.checked);
        });
    </script>
</body>
</html>